unit RSTrans;

/////////////////////////////////////////////////////////////////////////////
//                     Part of RS232 project                               //
//        Sending and receiving files through RS232 interface              //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Serial transmission management for that project                         //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

interface

uses
  Forms, SysUtils, Classes, CPort;

const
  SingleFile = 0;           // Sending a single file
  FirstFile = 1;            // Sending the first file of multiple choice
  MiddleFile = 2;           // Sending a file of multiple choice
  LastFile = 3;             // Sending the last file of multiple choice

type
  TTransCtrl = (tcNone, tcXonXoff, tcCtsRts);
  TStatusCode = (scConnected, scDisconnected, scBufferFull, scEndOfTransmission,
                 scSendFile, scReceiveFile, scAskForData, scWaitForSending,
                 scGetStartToSend, scGetXOFF, scGetXON, scSendXOFF, scSendXON);
  TStatusEvent  = procedure (Sender: TObject; StatusCode: TStatusCode) of object;
  TRxDataEvent  = procedure (Sender: TObject; Data: integer) of object;
  TProcessRxCodeEvent  = procedure (Sender: TObject; Code: integer) of object;
  TProcessTxCodeEvent  = procedure (Sender: TObject; Code: integer) of object;
  TProcessRxValueEvent = procedure (Sender: TObject; Value: integer) of object;
  TProcessTxValueEvent = procedure (Sender: TObject; Value: integer) of object;

  TMySerialCom = class(TComponent)
  private
    FComPort: TComPort;                 // Com port
    FFile: textfile;                    // File to send / save

    FMinBuffer: byte;                   // Minimum buffer level
    FBuffer: array [1..1024] of char;   // 1kb file read buffer
    FInBuffer: string;                  // Input buffer
    FOutBuffer: string;                 // Output buffer
    FTransControl: TTransCtrl;          // Flow control

    FReceiveState: boolean;             // Waiting to receive data
    FXoffState: boolean;                // Waiting for Xon
    FWaitingSignal: boolean;            // Waiting signal which ask transmission

    FStopped: boolean;            // Transmission stop requested by user or
                                  // after error
    FSendingMode: byte;           // SingleFile, FirstFile, MiddleFile, LastFile
    FNbTransmittedChar: longint;  // Number of already transmitted character

    FOnStatusChange: TStatusEvent;            // status change event
    FOnRxData: TRxDataEvent;                  // received data event
    FOnProcessRxCode: TProcessRxCodeEvent;    // processing received code event
    FOnProcessTxCode: TProcessTxCodeEvent;    // processing sended code event
    FOnProcessRxValue: TProcessRxValueEvent;  // processing received value event
    FOnProcessTxValue: TProcessTxValueEvent;  // processing sended value event

    function CodeEIA(AChar: char): char;
    procedure NoneCoding;
    procedure ISOCoding;
    procedure EIACoding;
    function DecodeEIA(AChar: char): char;
    procedure DoCoding;
    procedure NoneDecoding;
    procedure ISODecoding;
    procedure EIADecoding;
    function DoDecoding(Limit: integer): boolean;

    function FindCode(ACode: integer; AChar: char): boolean;
    procedure SendChar(AChar: char; IsCode: boolean);
    procedure SendNullChar(Quantity: integer);
    function SendRemoveLines(Quantity: integer): boolean;
    function FindSendStartCode(StartCode: integer): boolean;
    function ReceiveRemoveLines(var Quantity: integer): boolean;
    function FindReceiveStartCode(StartCode: integer): boolean;
    function FindReceiveEndCode(EndCode: integer): boolean;

    procedure ClearInOutBuffers;
    function SendFillInBuffer: boolean;
    procedure SendProcessBuffer;
    procedure SendRestOfBuffer;
    function ReceiveProcessBuffer: boolean;
    procedure ReceiveRestOfBuffer;

    procedure SendRxChar(AChar: char);
    procedure ReceiveRxChar(AChar: char);
    procedure ComPortRxChar(Sender: TObject; Count: Integer);

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property ComPort: TComPort read FComPort;
    property Stopped: boolean read FStopped write FStopped;
    property SendingMode: byte read FSendingMode write FSendingMode;
    property NbTransmittedChar: longint
      read FNbTransmittedChar write FNbTransmittedChar;

    property OnStatusChange: TStatusEvent read FOnStatusChange write FOnStatusChange;
    property OnRxData: TRxDataEvent read FOnRxData write FOnRxData;
    property OnProcessTxCode: TProcessTxCodeEvent read FOnProcessTxCode write FOnProcessTxCode;
    property OnProcessRxCode: TProcessRxCodeEvent read FOnProcessRxCode write FOnProcessRxCode;
    property OnProcessTxValue: TProcessTxValueEvent read FOnProcessTxValue write FOnProcessTxValue;
    property OnProcessRxValue: TProcessRxValueEvent read FOnProcessRxValue write FOnProcessRxValue;

    procedure SetupComPort;
    function OpenComPort: boolean;
    procedure CloseComPort;
    function SendFile(AFilename: TFilename): byte;
      // Return code: 0 No error, 1 File too small, 2 Can't find start code
    procedure ReceiveFile(AFilename: TFilename);
    procedure AskForData;
    procedure WaitForSending;
  end;

var
  SerialCom: TMySerialCom;

implementation

uses
  RSCfgmng, RSLog;

{ TMySerialCom }

{*****************************************************************************}
{ Coding                                                                      }
{*****************************************************************************}

function TMySerialCom.CodeEIA(AChar: char): char;
begin
  case AChar of
      #0  : Result := #0;
      #8  : Result := #42;
      #9  : Result := #62;
      #13 : Result := #128;
  { } #32 : Result := #16;
  {%} #37 : Result := #11;
  {&} #38 : Result := #14;
  {(} #40 : Result := #26;
  {)} #41 : Result := #74;
  {+} #43 : Result := #112;
  {,} #44 : Result := #59;
  {-} #45 : Result := #64;
  {.} #46 : Result := #107;
  {/} #47 : Result := #49;
  {0} #48 : Result := #32 ;
  {1} #49 : Result := #1;
  {2} #50 : Result := #2;
  {3} #51 : Result := #19;
  {4} #52 : Result := #4;
  {5} #53 : Result := #21;
  {6} #54 : Result := #22;
  {7} #55 : Result := #7;
  {8} #56 : Result := #8;
  {9} #57 : Result := #25;
  {A} #65 : Result := #97;
  {B} #66 : Result := #98;
  {C} #67 : Result := #115;
  {D} #68 : Result := #100;
  {E} #69 : Result := #117;
  {F} #70 : Result := #118;
  {G} #71 : Result := #103;
  {H} #72 : Result := #104;
  {I} #73 : Result := #121;
  {J} #74 : Result := #81;
  {K} #75 : Result := #82;
  {L} #76 : Result := #67;
  {M} #77 : Result := #84;
  {N} #78 : Result := #69;
  {O} #79 : Result := #70;
  {P} #80 : Result := #87;
  {Q} #81 : Result := #88;
  {R} #82 : Result := #73;
  {S} #83 : Result := #50;
  {T} #84 : Result := #35;
  {U} #85 : Result := #52;
  {V} #86 : Result := #37;
  {W} #87 : Result := #38;
  {X} #88 : Result := #55;
  {Y} #89 : Result := #56;
  {Z} #90 : Result := #41;
      #219: Result := #127;
       else Result := AChar;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.NoneCoding;
var
  AChar: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);
  FOutBuffer:= FOutBuffer + AChar;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ISOCoding;
var
  AChar: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);

  // not CR + LF
  if not((ord(AChar)=13) and (FInBuffer[1]=#10)) then
    FOutBuffer:= FOutBuffer + AChar;

  if length(FInBuffer)>0 then
  begin
    // 0
    if (ord(AChar)=10) and (FInBuffer[1]=#79) then
    begin
      delete(FInBuffer,1,1);
      FOutBuffer:= FOutBuffer + #58; // :
    end;
    // CR + LF
    if (ord(AChar)=13) and (FInBuffer[1]=#10) then
    begin
      delete(FInBuffer,1,1);
      FOutBuffer:= FOutBuffer + #10; // LF
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.EIACoding;
var
  AChar: char;
  Answer: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);
  Answer:= CodeEIA(AChar);
  if Answer=#128 then
  begin
    AChar:= FInBuffer[1];
    if AChar=#10 then
    begin
      delete(FInBuffer,1,1);
      Answer:= AChar;
    end;
  end;
  FOutBuffer:= FOutBuffer + Answer;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.DoCoding;
begin
  case CurrentCfg.Coding of
    Sans: NoneCoding;
    ISO : ISOCoding;
    EIA : EIACoding;
  end;
end;

{*****************************************************************************}
{ Decoding                                                                    }
{*****************************************************************************}

function TMySerialCom.DecodeEIA(AChar: char): char;
begin
  case AChar of
    #0  : Result := #0;
    #1  : Result := #49; {1}
    #2  : Result := #50; {2}
    #4  : Result := #52; {4}
    #7  : Result := #55; {7}
    #8  : Result := #56; {8}
    #11 : Result := #37; {%}
    #14 : Result := #38; {&}
    #16 : Result := #32; { }
    #19 : Result := #51; {3}
    #21 : Result := #53; {5}
    #22 : Result := #54; {6}
    #25 : Result := #57; {9}
    #26 : Result := #40; {(}
    #32 : Result := #48; {0}
    #35 : Result := #84; {T}
    #37 : Result := #86; {V}
    #38 : Result := #87; {W}
    #41 : Result := #90; {Z}
    #42 : Result := #8;
    #49 : Result := #47; {/}
    #50 : Result := #83; {S}
    #52 : Result := #85; {U}
    #55 : Result := #88; {X}
    #56 : Result := #89; {Y}
    #59 : Result := #44; {,}
    #62 : Result := #9;
    #64 : Result := #45; {-}
    #67 : Result := #76; {L}
    #69 : Result := #78; {N}
    #70 : Result := #79; {O}
    #73 : Result := #82; {R}
    #74 : Result := #41; {)}
    #81 : Result := #74; {J}
    #82 : Result := #75; {K}
    #84 : Result := #77; {M}
    #87 : Result := #80; {P}
    #88 : Result := #81; {Q}
    #97 : Result := #65; {A}
    #98 : Result := #66; {B}
    #100: Result := #68; {D}
    #103: Result := #71; {G}
    #104: Result := #72; {H}
    #107: Result := #46; {.}
    #112: Result := #43; {+}
    #115: Result := #67; {C}
    #117: Result := #69; {E}
    #118: Result := #70; {F}
    #121: Result := #73; {I}
    #127: Result := #219;
    #128: Result := #13;
    else Result := AChar;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.NoneDecoding;
var
  AChar: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);
  FOutBuffer:= FOutBuffer + AChar;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ISODecoding;
var
  AChar: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);
  if ord(AChar)=10 then
  begin
    // LF, so save CR + LF
    FOutBuffer:= FOutBuffer + #13 + AChar;
  end
  else
  begin
    // Ignore all CR
    if ord(AChar)<>13 then FOutBuffer:= FOutBuffer + AChar;
  end;

  if length(FInBuffer)>0 then
  begin
    if (ord(AChar)=10) then
    begin
      if (FInBuffer[1]=#58 {:}) then
      begin
        delete(FInBuffer,1,1);
        FOutBuffer:= FOutBuffer + #79 {O};
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.EIADecoding;
var
  AChar: char;
  Answer: char;
begin
  AChar:= FInBuffer[1];
  delete(FInBuffer,1,1);
  Answer:= DecodeEIA(AChar);
  FOutBuffer:= FOutBuffer + Answer;
  if Answer= #13 then FOutBuffer:= FOutBuffer + #10;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.DoDecoding(Limit: integer): boolean;
var
  Found: boolean;
begin
  Found:= false;
  while (length(FInBuffer) > Limit) do
  begin
    case CurrentCfg.Coding of
      Sans: NoneDecoding;
      ISO : ISODecoding;
      EIA : EIADecoding;
    end;
    // Find end code
    if CurrentCfg.ReceiveFindStartEnd then
      Found:= FindReceiveEndCode(StrToInt(CurrentCfg.EndCode));
    if Found then FInBuffer:= FInBuffer[1];
  end;
  Result:= Found;
end;

{*****************************************************************************}
{ Utilities                                                                   }
{*****************************************************************************}

function TMySerialCom.FindCode(ACode: integer; AChar: char): boolean;
begin
  Result:= ord(AChar) = ACode;
end;

{------------------------------------------------------------------------------}

{*****************************************************************************}
{ Send utilities                                                              }
{*****************************************************************************}

procedure TMySerialCom.SendChar(AChar: char; IsCode: boolean);
var
  Value: longint;
begin
  Application.ProcessMessages;
  if not FStopped then
  begin
    Value:= ord(AChar);
    if IsCode then
    begin
      if assigned(FOnProcessTxCode) then FOnProcessTxCode(self,Value);
    end
    else
    begin
      if assigned(FOnProcessTxValue) then FOnProcessTxValue(self,Value);
    end;
    FComPort.WriteStr(AChar);
    if (not FReceiveState) then
    begin
      inc(FNbTransmittedChar);
      // Xon / Xoff
      if (FTransControl = tcXonXoff) then
      repeat
        Application.ProcessMessages;
      until (not FXoffState) or FStopped;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.SendNullChar(Quantity: integer);
var
  N: integer;
begin
  if Quantity > 0 then
  begin
    N:= 0;
    while (N < Quantity) and not FStopped do
    begin
      SendChar(#00,true);
      inc(N);
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.SendRemoveLines(Quantity: integer): boolean;
var
  N: integer;
  Finished: boolean;
  AChar: char;
begin
  if Quantity > 0 then
  begin
    Finished:= false;
    N:= 0;
    while not (Finished or FStopped or eof(FFile)) do
    begin
      read(FFile,AChar);
      if Achar = #10 then inc(N);
      Finished:= (N = Quantity);
    end;
  end
  else Finished:= true;
  Result:= Finished;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.FindSendStartCode(StartCode: integer): boolean;
var
  Found: boolean;
  AChar: char;
begin
  Found:= false;
  while not (Found or FStopped or eof(FFile)) do
  begin
    read(FFile,AChar);
    Found:= FindCode(StartCode, AChar)
  end;
  Result:= Found;
end;

{*****************************************************************************}
{ Receive utilities                                                           }
{*****************************************************************************}

function TMySerialCom.ReceiveRemoveLines(var Quantity: integer): boolean;
var
  Finished: boolean;
  AChar: char;
begin
  if Quantity > 0 then
  begin
    Finished:= false;
    while not (Finished or FStopped or (length(FInBuffer) <= FMinBuffer)) do
    begin
      AChar:= FInBuffer[1];
      delete(FInBuffer,1,1);
      if Achar = #10 then dec(Quantity);
      Finished:= (Quantity = 0);
    end;
  end
  else Finished:= true;
  Result:= Finished;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.FindReceiveStartCode(StartCode: integer): boolean;
var
  Found: boolean;
  AChar: char;
begin
  Found:= false;
  if length(FInBuffer)>0 then
  begin
    AChar:= FInBuffer[1];
    case CurrentCfg.Coding of
      EIA : AChar:= DecodeEIA(AChar);
    end;
    Found:= FindCode(StartCode, AChar);
    if not Found then delete(FInBuffer,1,1);
  end;
  Result:= Found;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.FindReceiveEndCode(EndCode: integer): boolean;
var
  Found: boolean;
  AChar: char;
begin
  Found:= false;
  if length(FInBuffer)>0 then
  begin
    AChar:= FInBuffer[1];
    case CurrentCfg.Coding of
      EIA : AChar:= DecodeEIA(AChar);
    end;
    Found:= FindCode(EndCode, AChar);
  end;
  Result:= Found;
end;

{*****************************************************************************}
{ Buffer management                                                           }
{*****************************************************************************}

procedure TMySerialCom.ClearInOutBuffers;
begin
  FInBuffer:= '';
  FOutBuffer:= '';
end;

{------------------------------------------------------------------------------}

function TMySerialCom.SendFillInBuffer: boolean;
// Add character to buffer up to MiniBuffer
// Return true if end of transmisson found
var
  AChar: char;
begin
  Result:= false;
  while (length(FInBuffer) < FMinBuffer) and (not (FStopped or eof(FFile))) do
  begin
    read(FFile,AChar);
    // Find end code
    if CurrentCfg.SendFindStartEnd then
    begin
      Result:= FindCode(StrToInt(CurrentCfg.EndCode), AChar);
      if Result then
      begin
        if FSendingMode in [SingleFile, LastFile] then
          FInBuffer:= FInBuffer + AChar;
      end
      else
        FInBuffer:= FInBuffer + AChar;
    end
    else FInBuffer:= FInBuffer + AChar;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.SendProcessBuffer;
begin
  DoCoding;
  // Send
  while length(FOutBuffer) > FMinBuffer do
  begin
    SendChar(FOutBuffer[1],false);
    delete(FOutBuffer,1,1);
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.SendRestOfBuffer;
begin
  while length(FInBuffer) > 0 do DoCoding;
  // Send
  while length(FOutBuffer) > 0 do
  begin
    SendChar(FOutBuffer[1],false);
    delete(FOutBuffer,1,1);
  end;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.ReceiveProcessBuffer: boolean;
var
  AString: string;
  Stop: boolean;
begin
  Stop:= DoDecoding(FMinBuffer);
  AString:= '';
  while not (Stop or FStopped or (length(FOutBuffer) <= FMinBuffer)) do
  begin
    AString:= AString + FOutBuffer[1];
    delete(FOutBuffer,1,1);
  end;
  write(FFile,AString);
  Result:= Stop;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ReceiveRestOfBuffer;
begin
  DoDecoding(0);
  // Save
  write(FFile,FOutBuffer);
  FOutBuffer:= '';
end;

{*****************************************************************************}
{ ComPort configuration                                                       }
{*****************************************************************************}

procedure TMySerialCom.SendRxChar(AChar: char);
begin
  if FWaitingSignal then
  begin
    // Waiting signal before sending
      if FindCode(StrToInt(CurrentCfg.SendCode),AChar) then
      begin
        FWaitingSignal:= false;
        if assigned(FOnStatusChange) then FOnStatusChange(self,scGetStartToSend);
      end;
  end
  else
  begin
    // Xon Xoff while sending data
    if FTransControl = tcXonXoff then
    begin
        if FXoffState then
        begin
          FXoffState:= AChar <> FComPort.FlowControl.XonChar;
          if not FXoffState then
          begin
            if assigned(FOnStatusChange) then FOnStatusChange(self,scGetXon);
          end;
        end
        else
        begin
          FXoffState:= AChar = FComPort.FlowControl.XoffChar;
          if FXoffState then
          begin
            if assigned(FOnStatusChange) then FOnStatusChange(self,scGetXOFF);
          end;
        end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ReceiveRxChar(AChar: char);
begin
  FInBuffer:= FInBuffer + AChar;
  inc(FNbTransmittedChar,1);
  // Xon Xoff while receiving
  if FTransControl = tcXonXoff then
  begin
    if length(FInBuffer) > 2048 then
    begin
      SendChar(FComPort.FlowControl.XoffChar, true);
      if assigned(FOnStatusChange) then FOnStatusChange(self,scSendXOFF);
      FXoffState:= true;
    end
    else
    begin
      if FXoffState and (length(FInBuffer) < 50) then
      begin
        SendChar(FComPort.FlowControl.XonChar, true);
        if assigned(FOnStatusChange) then FOnStatusChange(self,scSendXON);
        FXoffState:= false;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ComPortRxChar(Sender: TObject; Count: Integer);
var
  AString: string;
  AChar: char;
  Value: longint;
  n: integer;
begin
  if (Count > 0) then
  begin
    FComPort.ReadStr(AString,count);
    // needed for cts/rts as it did not work if we take char by char
    for N:= 1 to length(AString) do
    begin
      AChar:= AString[N];
      if assigned(FOnProcessRxValue) then
      begin
        Value:= ord(AChar);
        FOnProcessRxValue(self,Value);
      end;

      if FReceiveState then ReceiveRxChar(AChar)
                       else SendRxChar(AChar);
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.SetupComPort;
begin
  if FComPort <> nil then
  with CurrentCfg do
  begin
    FComPort.BeginUpdate;
    FComPort.Port:= Port;
    FComPort.BaudRate:= StrToBaudRate(Speed);
    case Parity of
      1: FComPort.Parity.Bits:= prEven;
      2: FComPort.Parity.Bits:= prOdd;
    else
      FComPort.Parity.Bits:= prNone;
    end;
    FComPort.DataBits:= StrToDataBits(DataBits);
    FComPort.StopBits:= StrToStopBits(StopBits);
    case FlowControl of
      1: begin
           FTransControl:= tcCtsRts;
           FComPort.FlowControl.FlowControl:= fcHardware;
         end;
      2: begin
           FTransControl:= tcXonXoff;
           FComport.FlowControl.FlowControl:= fcNone;
         end;
    else
      FTransControl:= tcNone;
      FComport.FlowControl.FlowControl:= fcNone;
    end;
    FComPort.EndUpdate;
  end;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.OpenComPort: boolean;
begin
  with FComPort do
  begin
    if not Connected then Open;
    if assigned(FOnStatusChange) then
    begin
      if Connected then FOnStatusChange(self,scConnected)
                   else FOnStatusChange(self,scDisconnected);
    end;
    ClearBuffer(true,true);
    result:= Connected;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.CloseComPort;
begin
  with FComPort do
  begin
    if Connected then
    begin
      ClearBuffer(true,true);
      Close;
    end;
    if assigned(FOnStatusChange) then
    begin
      if Connected then FOnStatusChange(self,scConnected)
                   else FOnStatusChange(self,scDisconnected);
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TMySerialCom.SendFile(AFilename: TFilename): byte;
// Return code: 0 No error, 1 File too small, 2 Can't find start code
var
  Stop: boolean; // Transmission ended normally
begin
  if assigned(FOnStatusChange) then FOnStatusChange(self,scSendFile);
  Result:= 0;
  FReceiveState:= false;
  FXoffState:= false;
  if FileExists(AFilename) then
  begin
    assignfile(FFile,AFilename);
    SetTextBuf(FFile,FBuffer);
    reset(FFile);
    ClearInOutBuffers;
    try
      // Send start null char
      SendNullChar(StrToInt(CurrentCfg.StartNullChar));
      // Remove lines
      Stop:= not SendRemoveLines(StrToInt(CurrentCfg.SendRemoveLines));
      if not Stop then
      begin
        // Find start code
        if CurrentCfg.SendFindStartEnd then
        begin
          if not FindSendStartCode(StrToInt(CurrentCfg.StartCode)) then
          begin
            Stop:= true;
            FStopped:= true;
            Result:= 2;
          end;
          if ((not Stop) and (FSendingMode in [SingleFile, FirstFile])) then
            FInBuffer:= FInBuffer + chr(StrToInt(CurrentCfg.StartCode));
        end;
        // Process file
        while not(Stop or FStopped or eof(FFile)) do
        begin
          Stop:= SendFillInBuffer;
          SendProcessBuffer;
          Application.ProcessMessages;
        end;
        // Finish sending OutBuffer
        if not FStopped then SendRestOfBuffer;
        // Send end null char
        if not FStopped then SendNullChar(StrToInt(CurrentCfg.EndNullChar));
        close(FFile);
      end
      else
      begin
        close(FFile);
        FStopped:= true;
        Result:= 1;
      end;
    except
      close(FFile);
      FStopped:= true;
      raise;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.ReceiveFile(AFilename: TFilename);
var
  Found: boolean;
  Quantity: integer;
begin
  if assigned(FOnStatusChange) then FOnStatusChange(self,scReceiveFile);
  FReceiveState:= true;
  FXoffState:= false;
  try
    assignfile(FFile,AFilename);
    SetTextBuf(FFile,FBuffer);
    rewrite(FFile);
    ClearInOutBuffers;
    // Remove lines
    Quantity:= StrToInt(CurrentCfg.ReceiveRemoveLines);
    Found:= false;
    while not(FStopped or Found) do
    begin
      Found:= ReceiveRemoveLines(Quantity);
      Application.ProcessMessages;
    end;
    if not FStopped then
    begin
      // Find start code
      if CurrentCfg.ReceiveFindStartEnd then
      begin
        Found:= false;
        while not(FStopped or Found) do
        begin
          Found:= FindReceiveStartCode(StrToInt(CurrentCfg.StartCode));
          Application.ProcessMessages;
        end;
      end;
      // Process reception
      Found:= false;
      while not(Found or FStopped) do
      begin
        Found:= ReceiveProcessBuffer;
        Application.ProcessMessages;
      end;
      // Finish saving OutBuffer
      ReceiveRestOfBuffer;
    end;
    close(FFile);
  except
    close(FFile);
    FStopped:= true;
    raise;
  end;
  FReceiveState:= false;
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.AskForData;
var
  Code: char;
begin
  if assigned(FOnStatusChange) then FOnStatusChange(self,scAskForData);
  Code:= chr(StrToInt(CurrentCfg.SendCode));
  if Code = FComPort.FlowControl.XonChar then
    SendChar(FComPort.FlowControl.XoffChar, true);
  SendChar(Code,true);
end;

{------------------------------------------------------------------------------}

procedure TMySerialCom.WaitForSending;
var
  CurrentFlowControl: byte;
  Found: boolean;
begin
  if assigned(FOnStatusChange) then FOnStatusChange(self,scWaitForSending);
  // Change flow control to none in order to be able to detect any char (i.e. XON)
  Stopped:= false;
  CurrentFlowControl:= CurrentCfg.FlowControl;
  CurrentCfg.FlowControl:= 0;
  SetupComPort;
  if SerialCom.OpenComPort then
  begin
    Found:= false;
    FWaitingSignal:= true;
    while not(FStopped or Found) do
    begin
      Found:= FWaitingSignal = false;
      Application.ProcessMessages;
    end;
  end;
  CloseComPort;
  CurrentCfg.FlowControl:= CurrentFlowControl;
end;

{------------------------------------------------------------------------------}

constructor TMySerialCom.Create(AOwner: TComponent);
begin
  inherited;
  // Create and init comport
  FComPort := TComPort.Create(AOwner);
  with FComPort do
  begin
    Name := 'ComPort';
    BaudRate := br9600;
    Port := 'COM1';
    Parity.Bits := prNone;
    StopBits := sbOneStopBit;
    DataBits := dbEight;
    Events := [evRxChar, evTxEmpty, evRxFlag, evRing, evBreak, evCTS, evDSR, evError, evRLSD, evRx80Full];
    Flowcontrol.FlowControl:= fcNone;
    FlowControl.OutCTSFlow := False;
    FlowControl.OutDSRFlow := False;
    FlowControl.ControlDTR := dtrEnable;
    FlowControl.ControlRTS := rtsDisable;
    FlowControl.XonXoffOut := False;
    FlowControl.XonXoffIn := False;
  end;
  FComPort.OnRxChar:=  ComPortRxChar;
  FTransControl:= tcNone;
  // Init buffer
  FMinBuffer:= 4;
end;

{------------------------------------------------------------------------------}

destructor TMySerialCom.Destroy;
begin
  FComPort.Free;
  inherited;
end;

end.
