unit RSMain;

/////////////////////////////////////////////////////////////////////////////
//                     Part of RS232 project                               //
//        Sending and receiving files through RS232 interface              //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//                              Update history                             //
//                                                                         //
// 09/07/2000 version 0.9.0                                                //
//   First running version                                                 //
// 17/08/2000 version 0.9.1                                                //
//   Multi language                                                        //
// 18/08/2000 version 0.9.2                                                //
//   Correct trouble if directory stored in configuration did not exist.   //
//   Exception management during emitting or receiving.                    //
// 25/12/2000 version 0.9.3                                                //
//   Icon for windows recognized files.                                    //
// 26/05/2003 version 0.9.4                                                //
//   Build default language file if did not exist, correct ending if       //
//   comport cannot be open, on receiving, add extension corresponding to  //
//   the used filter if no extension                                       //
// 09/06/2003 version 0.9.5                                                //
//   Code cleanup to remove circular reference, own messages dialogs,      //
//   preference dialog, on sending multiple files with find start/end code //
//   remove intermdiate start/end codes, double click on file = view it   //
//   Lots of bugs solved.                                                  //
// 15/11/2003 version 0.9.6                                                //
//   Solve bug in start/stop code detection (must be on FInBuffer not      //
//   FOutBuffer)                                                           //
// 17/11/2003 version 0.9.7                                                //
//   Add waiting for signal (XON) to send data                             //
// 24/03/2007 version 0.9.8                                                //
//   Add logfile in all data transmission                                  //
//   Correct bug if transmission is cancel while waiting for send signal,  //
//     file was send anyway                                                //
// 24/06/2008 version 0.9.9                                                //
//   Correct bug in log function if file did not exist while we turn on log//
// 25/05/2011 version 0.9.9a                                               //
//   Add data transmission configuration in logfile                        //
//   Add machine list refresh after machine creation or deletion           //
//   Correct bug in machine configuration deletion (was not working)       //
//   Correct bug in CTS/RTS flow control, was stopping after around 1280   //
//     character at 600 buads                                              //
//   Correct bug in receive data logging, we never get the value           //
// 25/05/2011 version 0.9.9b                                               //
//   Correct bug in code selection for code send to request transmission   //
//   Prevent deletion of currently used configuration                      //
//   Add import/export configuration facilities                            //
//   When reading configuration, if working directory did not exist, change//
//     it to application directory                                         //
// 22/08/2011 version 1.0.0                                                //
//   As 0.9.9 seems ok, change to a proper version number                  //
// 28/02/2012 version 1.0.1                                                //
//   Add RS232 software version in log file                                //
// 22/12/2012 version 1.0.2                                                //
//   Change transmitted character title depending if we send or receive    //
//   Add signal led's for information                                      //
//   Correct language file generation and some translations                //
// 08/05/2013 version 1.1.0                                                //
//   Change configuration file storage folder to use Windows way of doing  //
//     by using user application data folder. Storage place is displayed   //
//     in About box. If proper application data folder did not exist (zip  //
//     file instead of setup.exe) or if -L is in command line parameters,  //
//     configuration files are stored in same folder as exe                //
//   Add possibility to let serial port open, needed for some equipment    //
//     like MOXA ethernet to serial signal converter                       //
//   Correct configuration file management to allow addition of new        //
//     parameters                                                          //
// 24/11/2013 version 1.2.0                                                //
//   Add thumbnail display if a bmp, png or jpg picture exist with the     //
//     same filename                                                       //
// 16/12/2013 version 1.2.1                                                //
//   Thumbnail display can be switch on and off from preference.           //
//   Max thumbnail size defined in preference screen                       //
// 31/12/2014 version 1.2.2                                                //
//   Fix many grammatical, typographic and orthographic mistake in french  //
//     language file creation and in help                                  //
// 18/08/2015 version 1.2.3                                                //
//   Fix some grammatical, typographic and orthographic mistake in french  //
//     language file creation and in help                                  //
/////////////////////////////////////////////////////////////////////////////
//              Autre composants utiliss / Other components used          //
//  ComPort: Dejan Crnila  http://sourceforge.net/projects/comport         //
//                                                                         //
//                                                                         //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             }
{***************************************************************************}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Menus, ImgList, ActnList, StdCtrls, Buttons, ComCtrls, FileCtrl,
  ToolWin, ExtCtrls, ShellApi,MMSystem,
  RSTrans, CPortCtl, CPort;

type
  TListItemType = (itFile, itFolder);

  TFMain = class(TForm)
    SbrMain: TStatusBar;
    PnlPC: TPanel;
    TbrPC: TToolBar;
    TbnSpacer1: TToolButton;
    DcbPCDriveList: TDriveComboBox;
    TbnSpacer2: TToolButton;
    TbnPreviousDir: TToolButton;
    TbnSpacer3: TToolButton;
    TbnNewDir: TToolButton;
    TbnSpacer4: TToolButton;
    TbnSmallIcon: TToolButton;
    TbnList: TToolButton;
    TbnDetail: TToolButton;
    PnlPCDirectory: TPanel;
    PnlPCFileList: TPanel;
    LvFileList: TListView;
    PnlPCTitle: TPanel;
    PnlMachine: TPanel;
    LblMachine: TLabel;
    SbnConfig: TSpeedButton;
    LblFollowup: TLabel;
    LblTransmittedCharTitle: TLabel;
    LblTransmittedChar: TLabel;
    PnlMachineTitle: TPanel;
    CbxMachine: TComboBox;
    MemFollowup: TMemo;
    BbnCancel: TBitBtn;
    BbnSend: TBitBtn;
    BbnReceive: TBitBtn;
    MnuMain: TMainMenu;
    MnuFile: TMenuItem;
    N1: TMenuItem;
    MnuFileQuit: TMenuItem;
    MnuTools: TMenuItem;
    MnuToolsConfig: TMenuItem;
    MnuLanguage: TMenuItem;
    Francais1: TMenuItem;
    MnuToolsSaveStatus: TMenuItem;
    MnuHelp: TMenuItem;
    MnuHelpChapter: TMenuItem;
    MnuHelpFind: TMenuItem;
    N2: TMenuItem;
    MnuHelpABout: TMenuItem;
    ActionList: TActionList;
    ActFileQuit: TAction;
    ActHelpAbout: TAction;
    ActConfigSerial: TAction;
    ActReadDir: TAction;
    ActPreviousDir: TAction;
    ActNewDir: TAction;
    ActDriveChange: TAction;
    ActSmallIcon: TAction;
    ActList: TAction;
    ActDetail: TAction;
    ActSend: TAction;
    ActReceive: TAction;
    ActCancel: TAction;
    IlMain: TImageList;
    PopMnuPCFileList: TPopupMenu;
    PmnuRefresh: TMenuItem;
    DlgSave: TSaveDialog;
    IlFileIcon: TImageList;
    ActHelpChapter: TAction;
    ActHelpFind: TAction;
    ActChangeMachine: TAction;
    ActEditMachine: TAction;
    TmrTransmittedChar: TTimer;
    ToolButton1: TToolButton;
    TbnRefresh: TToolButton;
    ActViewFile: TAction;
    PmnuDisplay: TMenuItem;
    ActPreference: TAction;
    PnlInfo: TPanel;
    ComLedConn: TComLed;
    ComLedRI: TComLed;
    ComLedCD: TComLed;
    ComLedDSR: TComLed;
    LblRI: TLabel;
    LblCD: TLabel;
    LblConn: TLabel;
    LblDSR: TLabel;
    ComLedCTS: TComLed;
    LblCTS: TLabel;
    LblTx: TLabel;
    ComLedTx: TComLed;
    LblRx: TLabel;
    ComLedRx: TComLed;
    ActDisconnect: TAction;
    BbnDisconnect: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ActFileQuitExecute(Sender: TObject);
    procedure ActConfigSerialExecute(Sender: TObject);
    procedure ActPreferenceExecute(Sender: TObject);
    procedure ActHelpAboutExecute(Sender: TObject);
    procedure ActHelpChapterExecute(Sender: TObject);
    procedure ActHelpFindExecute(Sender: TObject);
    procedure ActReadDirExecute(Sender: TObject);
    procedure ActPreviousDirExecute(Sender: TObject);
    procedure ActNewDirExecute(Sender: TObject);
    procedure ActDriveChangeExecute(Sender: TObject);
    procedure ActSmallIconExecute(Sender: TObject);
    procedure ActListExecute(Sender: TObject);
    procedure ActDetailExecute(Sender: TObject);
    procedure ActViewFileExecute(Sender: TObject);
    procedure ActChangeMachineExecute(Sender: TObject);
    procedure ActEditMachineExecute(Sender: TObject);procedure LvFileListColumnClick(Sender: TObject; Column: TListColumn);
    procedure ActReceiveExecute(Sender: TObject);
    procedure ActSendExecute(Sender: TObject);
    procedure ActCancelExecute(Sender: TObject);
    procedure LvFileListCompare(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer);
    procedure LvFileListDblClick(Sender: TObject);
    procedure LvFileListEdited(Sender: TObject; Item: TListItem;
      var S: String);
    procedure LvFileListEditing(Sender: TObject; Item: TListItem;
      var AllowEdit: Boolean);
    procedure LvFileListKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure TmrTransmittedCharTimer(Sender: TObject);
    procedure ActDisconnectExecute(Sender: TObject);
    procedure LvFileListMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
  private
    { Dclarations prives }
    ColumnToSort: integer;
    CurrentFilename: TFilename;
    OldFilename: TFilename;
    LogFilename: TFilename;
    StartTime: longword;             // time when transmission start
    procedure AddNewItem(Item: TListItem; AData: TWin32FindData);
    function GetListItemType(Item: TListItem): TListItemType;
    procedure ClearLanguageMenu;
    procedure MnuLanguageClick(Sender: TObject);
    procedure BuildLanguageMenu;
    procedure InitAppli;
    procedure GotoDir(ADirectory: string);
    procedure UpdateGUI(Transmitting: boolean);
    procedure CleanMemFollowup;
    procedure SetLogCfg;

    procedure SerialStatusChange(Sender: TObject; StatusCode: TStatusCode);
    procedure SerialComProcessRxCode(Sender: TObject; Code: Integer);
    procedure SerialComProcessTxCode(Sender: TObject; Code: Integer);
    procedure SerialComProcessRxValue(Sender: TObject; Value: Integer);
    procedure SerialComProcessTxValue(Sender: TObject; Value: Integer);
  public
    { Dclarations publiques }
    procedure ShowHint(Sender: TObject);

  end;

var
  FMain: TFMain;

implementation

{$R *.dfm}

uses
  RSCommon, RSAbout, RSLng, RSLog, RSCfgmng, RSConfig, RSPref, RSMsg,
  RSView, RSThumb;

{*****************************************************************************}
{ Form function                                                               }
{*****************************************************************************}

procedure TFMain.ShowHint(Sender: TObject);
begin
  SbrMain.SimplePanel:= true;
  SbrMain.SimpleText:= Application.hint;
end;

{------------------------------------------------------------------------------}

procedure TFMain.FormCreate(Sender: TObject);
var
  SFI: TSHFileInfo;
  AppliName: string;
begin
  // Get files icon image list
  IlFileIcon.Handle := SHGetFileInfo('', 0, SFI, SizeOf(SFI),
   SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
  // Get application directory
  AppliDir:= ExtractFilePath(Application.ExeName);
  // Get application name
  AppliName:= ChangeFileExt(ExtractFileName(Application.ExeName),'');
  // Get configuration directory and create it
  AppliCfgDir:= GetAppdataFolder;
  if (AppliCfgDir = '') or ((ParamCount > 0) and (FindCmdLineSwitch('L'))) then
    AppliCfgDir:= AppliDir
  else
  begin
    AppliCfgDir:= AppliCfgDir + PathDelim + 'AJS' + PathDelim + AppliName + PathDelim;
    if (not DirectoryExists(AppliCfgDir)) then  AppliCfgDir:= AppliDir;
  end;
  // Get application configuration filename
  AppliCfgFilename:= AppliCfgDir + AppliName + '.ini';
  // Get log filename
  LogFilename:= AppliCfgDir + AppliName + '.log';
  // Create application configuration storage
  AppliCfg:= TAppliCfg.create;
  // Create default application configuration file if needed
  if not FileExists(AppliCfgFilename) then AppliCfg.CreateDefault(AppliCfgFilename);
  // Load application configuration
  AppliCfg.LoadFromFile(AppliCfgFilename);
  // Create default language file if needed
  if not ExistLanguage(RefLng) then CreateLanguageFile;
  if AppliCfg.Language = '' then AppliCfg.Language:= RefLng;
  // Build language menu
  BuildLanguageMenu;
  // Create log component
  Log:= TRSLog.Create(Self);
  //Set log file configuration
  setLogCfg;
  // Get serial communication configuration filename
  SerialCfgFilename:= ChangeFileExt(AppliCfgFilename,SerialCfgExt);
  // Create default serial communication configuration file if needed
  if not FileExists(SerialCfgFilename) then CreateSerialCfgFile;
  // Create serial configuration storage
  CurrentCfg:= TCfgSerial.Create;
  TmpCfg:= TCfgSerial.Create;
  // Create serialcom component
  SerialCom:= TMySerialCom.Create(Self);
  // Associzte signals leds
  ComLedConn.ComPort:= SerialCom.ComPort;
  ComLedRI.ComPort:= SerialCom.ComPort;
  ComLedCD.ComPort:= SerialCom.ComPort;
  ComLedDSR.ComPort:= SerialCom.ComPort;
  ComLedCTS.ComPort:= SerialCom.ComPort;
  ComLedTx.ComPort:= SerialCom.ComPort;
  ComLedRx.ComPort:= SerialCom.ComPort;
  // Initialise application
  InitAppli;
end;

{------------------------------------------------------------------------------}

procedure TFMain.FormShow(Sender: TObject);
begin
  BbnDisconnect.Top:= BbnCancel.Top; 
  Application.OnHint:= ShowHint;
  SbnConfig.Caption:= '';
  LblTransmittedCharTitle.Visible:= false;
  Tag:= 1; // Form is displayed
  LvFileList.SetFocus;

  SerialCom.OnStatusChange:= SerialStatusChange;
  SerialCom.OnProcessRxCode:= SerialComProcessRxCode;
  SerialCom.OnProcessTxCode:= SerialComProcessTxCode;
  SerialCom.OnProcessRxValue:= SerialComProcessRxValue;
  SerialCom.OnProcessTxValue:= SerialComProcessTxValue;
  Log.Filename:= LogFilename;
  Log.Reset;
end;

{------------------------------------------------------------------------------}

procedure TFMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  SerialCom.Stopped:= true;
  // Save current configuration
  AppliCfg.Machine:= CbxMachine.Text;
  AppliCfg.MainWindow:= BoundsRect;
  case LvFileList.ViewStyle of
    vsSmallIcon: AppliCfg.FileListDisplay:= 0;
    vsList: AppliCfg.FileListDisplay:= 1;
    vsReport: AppliCfg.FileListDisplay:= 2;
  else
    AppliCfg.FileListDisplay:= 0;
  end;
  AppliCfg.SaveToFile(AppliCfgFilename);
end;

{------------------------------------------------------------------------------}

procedure TFMain.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  TmpCfg.Free;
  CurrentCfg.Free;
  SerialCom.Free;
  Log.Free;
  AppliCfg.Free;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActFileQuitExecute(Sender: TObject);
begin
  close;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActConfigSerialExecute(Sender: TObject);
begin
  with TFConfig.Create(Self) do
  try
    CopySerialCfg(CurrentCfg, TmpCfg);
    if Execute(TmpCfg,true) then
    begin
      if CurrentCfg.ConfigName = TmpCfg.ConfigName then
        CopySerialCfg(TmpCfg, CurrentCfg);
    end;
  finally
    GetAllConfigName(SerialCfgFilename,CbxMachine.Items);
    Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActPreferenceExecute(Sender: TObject);
begin
  with TFPreference.Create(Self) do
  try
    Execute;
    SetLogCfg;
    PnlInfo.Visible:=AppliCfg.InfoSignals;
  finally
    Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActHelpAboutExecute(Sender: TObject);
var
  FrmAbout: TFAbout;
begin
  FrmAbout:= TFAbout.Create(self);
  try
    FrmAbout.Showmodal;
  finally
    FrmAbout.Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActHelpChapterExecute(Sender: TObject);
const
  HELP_TAB= 15;
  CONTENTS_ACTIVE= -3;
begin
  Application.HelpCommand(HELP_TAB,CONTENTS_ACTIVE);
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActHelpFindExecute(Sender: TObject);
const
  HELP_TAB= 15;
  CONTENTS_ACTIVE= -2;
begin
  Application.HelpCommand(HELP_TAB,CONTENTS_ACTIVE);
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActReadDirExecute(Sender: TObject);
var
  srList: TSearchRec;
begin
  Screen.Cursor:= crHourGlass;
  LvFileList.Items.BeginUpdate;
  LvFileList.Items.clear;
  try
    if findfirst(AddBackSlash(CurrentDir)+'*.*', faAnyFile, srList)=0 then
    begin
      try
        repeat
          if not ((srList.Name='..') or (srList.Name='.')) then
            AddNewItem(LvFileList.Items.Add,srList.FindData);
        until findnext(srList) <> 0;
      finally
        Sysutils.findclose(srList);
      end;
    end;
  except
    on exception do;
  end;
  LvFileList.Items.EndUpdate;
  LvFileList.AlphaSort;
  PnlPCDirectory.Caption:= '  ' + CurrentDir;
  TbnPreviousDir.Enabled:= length(CurrentDir) > 3;
  Screen.Cursor:= crDefault;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActPreviousDirExecute(Sender: TObject);
begin
  if length(CurrentDir) > 3 then
  begin
    CurrentDir:= PCPreviousDir(CurrentDir);
    ActReadDir.Execute;
  end
  else beep;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActNewDirExecute(Sender: TObject);
const
  Name= 'Nouveau dossier';
var
  NewItem: TListItem;
  srFile: TSearchRec;
begin
///////////////////////////////////////////////////////////////////////////////
  if CreateDir(CurrentDir + Name) then
  begin
    NewItem:= LvFileList.Items.Add;
    try
      if findfirst(AddBackSlash(CurrentDir)+Name, faAnyFile, srFile)=0 then
      begin
        try
          if not ((srFile.Name='..') or (srFile.Name='.')) then
               AddNewItem(NewItem,srFile.FindData);
        finally
          Sysutils.findclose(srFile);
        end;
      end;
    except
      on exception do;
    end;
    NewItem.EditCaption;
  end
  else
  begin
    Beep;
    MessageDlg('Impossible de crer le rpertoire', mtError, [mbOk], 0);
  end;
///////////////////////////////////////////////////////////////////////////////
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActDriveChangeExecute(Sender: TObject);
var
  DriveNumber: integer;
  Drive: char;
  TmpDrive: char;
begin
///////////////////////////////////////////////////////////////////////////////
  TmpDrive:= DcbPCDriveList.Drive;
  if TmpDrive <> CurrentDrive then
  begin
    DriveNumber:= ord(TmpDrive) - ord('a') + 1;
    getdir(DriveNumber,CurrentDir);
    Drive:= (lowercase(CurrentDir))[1];
    if Drive = TmpDrive then
    begin
      CurrentDir:= AddBackSlash(CurrentDir);
      ActReadDir.Execute;
      CurrentDrive:= Drive;
    end
    else
    begin
      Beep;
      MessageDlg('Lecteur "' + TmpDrive + '" non disponible', mtError, [mbOk], 0);
      DcbPCDriveList.Drive:= CurrentDrive;
    end;
  end;
  if FMain.Tag = 1 then LvFileList.SetFocus;
///////////////////////////////////////////////////////////////////////////////
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActSmallIconExecute(Sender: TObject);
begin
  LvFileList.ViewStyle:= vsSmallIcon;
  TbnSmallIcon.Down:= true;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActListExecute(Sender: TObject);
begin
  LvFileList.ViewStyle:= vsList;
  TbnList.Down:= true;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActDetailExecute(Sender: TObject);
begin
  LvFileList.ViewStyle:= vsReport;
  TbnDetail.Down:= true;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActViewFileExecute(Sender: TObject);
begin
  if assigned(LvFileList.Selected) and
    (GetListItemType(LvFileList.Selected) = itFile) then
  begin
    with TFView.Create(Self) do
    try
      Execute(CurrentDir + LvFileList.Selected.Caption);
    finally
      Free;
      ActReadDir.Execute;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActChangeMachineExecute(Sender: TObject);
begin
  CurrentCfg.LoadFromFile(SerialCfgFilename,CbxMachine.Text);
  GotoDir(CurrentCfg.WorkingDir);
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActEditMachineExecute(Sender: TObject);
begin
  with TFConfig.Create(self) do
  try
    if Execute(CurrentCfg,false) then
    begin
      if CurrentDir <> AddBackSlash(CurrentCfg.WorkingDir) then
        GotoDir(CurrentCfg.WorkingDir);
    end;
  finally
    Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListColumnClick(Sender: TObject;
  Column: TListColumn);
begin
  ColumnToSort := Column.Index;
  (Sender as TCustomListView).AlphaSort;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActReceiveExecute(Sender: TObject);
var
  Answer: boolean;
  CfgString: string;
begin
  // Select save filename
  DlgSave.InitialDir:= CurrentDir;
  repeat
    Answer:= DlgSave.Execute;
    if Answer then
    begin
      CurrentFilename:= DlgSave.Filename;
      if FileExists(CurrentFilename) then
        Answer:= ShowYesNo(Self,GetMsg(251),[CurrentFilename]) = mrYes;
    end
    else
    begin
      CurrentFilename:= '';
      Answer:= true;
    end;
  until Answer;

  // Waiting received char
  if CurrentFilename <> '' then
  begin
    LblTransmittedCharTitle.Caption:= GetMsg(260);
    LblTransmittedCharTitle.Visible:= true;
    CleanMemFollowup;
    Log.Write(0, 0, '-------------------------------------------', ltInfo);
    CfgString := 'General: ' + CurrentCfg.ConfigName + ', ';
    CfgString := CfgString + CurrentCfg.WorkingDir;
    CfgString := CfgString + CurrentCfg.StartCode + ', ';
    CfgString := CfgString  + CurrentCfg.EndCode + ', ';
    CfgString := CfgString + CurrentCfg.SendCode;
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Serial port: ' + IntToStr(CurrentCfg.Coding) + ', ';
    CfgString := CfgString + CurrentCfg.Port + ', ';
    CfgString := CfgString + CurrentCfg.Speed + ', ';
    CfgString := CfgString + IntToStr(CurrentCfg.Parity) + ', ';
    CfgString := CfgString + CurrentCfg.DataBits + ', ';
    CfgString := CfgString + CurrentCfg.StopBits + ', ';
    CfgString := CfgString + IntToStr(CurrentCfg.FlowControl);
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Send: ' + CurrentCfg.StartNullChar + ', ' ;
    CfgString := CfgString + CurrentCfg.EndNullChar + ', ';
    if CurrentCfg.MessageBefore then CfgString := CfgString + 'yes, '
                                else CfgString := CfgString + 'no, ';
    CfgString := CfgString + CurrentCfg.SendRemoveLines + ', ';
    if CurrentCfg.SendFindStartEnd then CfgString := CfgString + 'yes, '
                                   else CfgString := CfgString + 'no, ';
    if CurrentCfg.SendWaitSignal then CfgString := CfgString + 'yes'
                                 else CfgString := CfgString + 'no';
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Receive: ' + CurrentCfg.ReceiveRemoveLines + ', ';
    if CurrentCfg.ReceiveFindStartEnd then CfgString := CfgString + 'yes, '
                                      else CfgString := CfgString + 'no, ';
    if CurrentCfg.ReceiveWaitSignal then CfgString := CfgString + 'yes'
                                    else CfgString := CfgString + 'no';
    Log.Write(0, 0, CfgString, ltInfo);
    Log.Write(0, 0, '-------------------------------------------', ltInfo);
    StartTime:= timeGetTime;
    SerialCom.Stopped:= false;
    SerialCom.SetupComPort;
    if SerialCom.OpenComPort then
    begin
      UpdateGUI(true);
      MemFollowUp.Lines.Add(Format(GetMsg(237),[CurrentFilename]));
      // Request opposite side to emit
      if CurrentCfg.ReceiveWaitSignal then SerialCom.AskForData;
      // Receive data
      SerialCom.ReceiveFile(CurrentFilename);
      if SerialCom.Stopped then MemFollowUp.Lines.Add(GetMsg(239))
                           else MemFollowUp.Lines.Add(GetMsg(238));
      UpdateGUI(false);
      SerialCom.CloseComPort;
      ActReadDir.Execute;
      Log.Write(0, 0, '-------------------------------------------', ltInfo);
      Log.Write(0, 0, '', ltInfo);
      Log.Save;
    end;
    LblTransmittedCharTitle.Visible:= false;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActSendExecute(Sender: TObject);
var
  Choice: TListItem;
  NextChoice: TListItem;
  ErrorCode: byte;
  CfgString: string;
begin
  if LvFileList.SelCount > 0 then
  begin
    LblTransmittedCharTitle.Caption:= GetMsg(240);
    LblTransmittedCharTitle.Visible:= true;
    CleanMemFollowup;
    Log.Write(0, 0, '-------------------------------------------', ltInfo);
    CfgString := 'General: ' + CurrentCfg.ConfigName + ', ';
    CfgString := CfgString + CurrentCfg.WorkingDir;
    CfgString := CfgString + CurrentCfg.StartCode + ', ';
    CfgString := CfgString  + CurrentCfg.EndCode + ', ';
    CfgString := CfgString + CurrentCfg.SendCode;
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Serial port: ' + IntToStr(CurrentCfg.Coding) + ', ';
    CfgString := CfgString + CurrentCfg.Port + ', ';
    CfgString := CfgString + CurrentCfg.Speed + ', ';
    CfgString := CfgString + IntToStr(CurrentCfg.Parity) + ', ';
    CfgString := CfgString + CurrentCfg.DataBits + ', ';
    CfgString := CfgString + CurrentCfg.StopBits + ', ';
    CfgString := CfgString + IntToStr(CurrentCfg.FlowControl);
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Send: ' + CurrentCfg.StartNullChar + ', ' ;
    CfgString := CfgString + CurrentCfg.EndNullChar + ', ';
    if CurrentCfg.MessageBefore then CfgString := CfgString + 'yes, '
                                else CfgString := CfgString + 'no, ';
    CfgString := CfgString + CurrentCfg.SendRemoveLines + ', ';
    if CurrentCfg.SendFindStartEnd then CfgString := CfgString + 'yes, '
                                   else CfgString := CfgString + 'no, ';
    if CurrentCfg.SendWaitSignal then CfgString := CfgString + 'yes'
                                 else CfgString := CfgString + 'no';
    Log.Write(0, 0, CfgString, ltInfo);
    CfgString := 'Receive: ' + CurrentCfg.ReceiveRemoveLines + ', ';
    if CurrentCfg.ReceiveFindStartEnd then CfgString := CfgString + 'yes, '
                                      else CfgString := CfgString + 'no, ';
    if CurrentCfg.ReceiveWaitSignal then CfgString := CfgString + 'yes'
                                    else CfgString := CfgString + 'no';
    Log.Write(0, 0, CfgString, ltInfo);
    Log.Write(0, 0, '-------------------------------------------', ltInfo);
    StartTime:= timeGetTime;
    SerialCom.Stopped:= false;
    if CurrentCfg.SendWaitSignal then SerialCom.WaitForSending;
    if not SerialCom.Stopped then
    begin
      SerialCom.SetupComPort;
      if SerialCom.OpenComPort then
      begin
        UpdateGUI(true);
        if CurrentCfg.MessageBefore then  ShowInfo(Self,GetMsg(231),[]);
        Choice:= LvFileList.Selected;
        if LvFileList.SelCount = 1 then SerialCom.SendingMode:= SingleFile
                                   else SerialCom.SendingMode:= FirstFile;
        // Process selection
        while (LvFileList.SelCount>0) and (not SerialCom.Stopped) do
        begin
          if LvFileList.SelCount>0 then
              NextChoice:= LvFileList.GetNextItem(Choice,sdAll,[isSelected])
          else NextChoice:= nil;
          CurrentFilename:= Choice.Caption;
          MemFollowUp.Lines.Add(Format(GetMsg(234),[CurrentFilename]));
          ErrorCode:= SerialCom.SendFile(CurrentDir + CurrentFilename);
          if SerialCom.Stopped then MemFollowUp.Lines.Add(GetMsg(236))
                               else MemFollowUp.Lines.Add(GetMsg(235));
          case ErrorCode of
            1: ShowError(Self,GetMsg(232),[CurrentFilename]);
            2: ShowError(Self,GetMsg(233),[CurrentCfg.StartCode + ' (' +
              Chr(StrToInt(CurrentCfg.StartCode))+ ')']);
          end;
          LvFileList.Items.Item[LvFileList.Items.IndexOf(Choice)].Selected:= false;
          LvFileList.Update;
          if NextChoice <> nil then
          begin
            if LvFileList.SelCount = 1 then SerialCom.SendingMode:= LastFile
                                       else SerialCom.SendingMode:= MiddleFile;
            Choice:= NextChoice;
          end;
        end;
        if (not CurrentCfg.StayOpen) then SerialCom.CloseComPort;
        ActDisconnect.Visible:= true;
      end;
    end;
    UpdateGUI(false);
    Log.Write(0, 0, '-------------------------------------------', ltInfo);
    Log.Write(0, 0, '', ltInfo);
    Log.Save;
    LblTransmittedCharTitle.Visible:= false;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActCancelExecute(Sender: TObject);
begin
  Log.Write(0, 0, '-> Stop requested by user', ltInfo);
  SerialCom.Stopped:= true;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ActDisconnectExecute(Sender: TObject);
begin
  Log.Write(0, 0, '-> Disconnection requested by user', ltInfo);
  if SerialCom.ComPort.Connected then SerialCom.CloseComPort;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
var
  S1: string;
  S2: string;
  Date1: TDateTime;
  Date2: TDateTime;
begin
  case ColumnToSort of
    0 : begin
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            Compare := CompareText(Item1.Caption, Item2.Caption);
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
    1 : begin
          S1:= trim(Item1.SubItems[0]);
          S2:= trim(Item2.SubItems[0]);
          if S1 = '' then Compare := -1
          else
          if S2 = '' then Compare := 1
          else
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            if StrToInt(S1) = StrToInt(S2) then
            begin
              Compare := CompareText(Item1.Caption, Item2.Caption);
            end
            else
            begin
              if StrToInt(S1) < StrToInt(S2) then Compare := -1
                                             else Compare := 1;
            end;
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
    2 : begin
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            Date1:= StrToDateTime(Item1.SubItems[1]);
            Date2:= StrToDateTime(Item2.SubItems[1]);
            if Date1=Date2 then
            begin
              Compare := CompareText(Item1.Caption, Item2.Caption);
            end
            else
            begin
              if Date1 < Date2 then Compare:=-1
                               else Compare:= 1;
            end;
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListDblClick(Sender: TObject);
begin
  if assigned(LvFileList.Selected) then
  begin
    case GetListItemType(LvFileList.Selected) of
      itFolder: begin
                  CurrentDir:= AddBackSlash(CurrentDir + LvFileList.Selected.Caption);
                  ActReadDir.Execute;
                end;
      itFile  : begin
                  ActViewFile.Execute;
                end ;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListEdited(Sender: TObject; Item: TListItem;
  var S: String);
begin
  if not renamefile(CurrentDir+OldFilename,CurrentDir+S) then
  begin
    Beep;
    S:= OldFilename;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListEditing(Sender: TObject; Item: TListItem;
  var AllowEdit: Boolean);
begin
  OldFilename:= LvFileList.Selected.Caption;
end;

{------------------------------------------------------------------------------}

procedure TFMain.LvFileListKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  Choice: TListItem;
  NextChoice: TListItem;
  Answer: word;
begin
  Answer:= mrNo;
  if (Key=VK_Delete) and (LvFileList.SelCount>0) then
  begin
    Choice:= LvFileList.Selected;
    while LvFileList.SelCount>0 do
    begin
      if LvFileList.SelCount>0 then
        NextChoice:= LvFileList.GetNextItem(Choice,sdAll,[isSelected])
        else NextChoice:= nil;
/////////////////////////////////////////////////////////////////////////////// // Affichage
      if Answer <> mrAll then
        Answer:= MessageDlg('Supprimer : ' + Choice.Caption, mtWarning,
           [mbYes,mbNo,mbAll], 0);
      if Answer in [mrYes,mrAll] then
      begin
        LvFileList.Items.BeginUpdate;
        case GetListItemType(Choice) of
          itFolder  : if RemoveDir(CurrentDir + Choice.Caption) then
                        LvFileList.Items.Delete(LvFileList.Items.IndexOf(Choice))
                      else
                      begin
                        Beep;
                        MessageDlg('Impossible de supprimer ce rpertoire.' +
                        #13 + 'Rpertoire non vide ou protg',mtError, [mbOk], 0); 
                      end;

          itFile    : if DeleteFile(CurrentDir + Choice.Caption) then
                        LvFileList.Items.Delete(LvFileList.Items.IndexOf(Choice))
                      else
                      begin
                        Beep;
                        MessageDlg('Impossible de supprimer ce fichier',mtError, [mbOk], 0);
                      end;
        end;
///////////////////////////////////////////////////////////////////////////////
        LvFileList.Items.EndUpdate;
        LvFileList.Update;
      end
      else LvFileList.Items.Item[LvFileList.Items.IndexOf(Choice)].Selected:= false;
      if NextChoice <> nil then Choice:= NextChoice;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.TmrTransmittedCharTimer(Sender: TObject);
begin
  if SerialCom.NbTransmittedChar > 0 then
    LblTransmittedChar.Caption:= IntToStr(SerialCom.NbTransmittedChar)
  else
    LblTransmittedChar.Caption:= '';
end;

{*****************************************************************************}
{ Private function                                                            }
{*****************************************************************************}


procedure TFMain.AddNewItem(Item: TListItem; AData: TWin32FindData);
var
  NewWindowsData: pWin32FindData;
  SFI: TSHFileInfo;
begin
  with Item do
  begin
    new(NewWindowsData);
    move(AData, NewWindowsData^, sizeof(AData));
    Data := NewWindowsData;
    // Filename
    Caption := AData.cFileName;
    // Icon
    SFI := GetShellInfo(
      AddBackSlash(CurrentDir) + Caption, AData.dwFileAttributes);
    ImageIndex := SFI.iIcon;

    if (AData.dwFileAttributes and File_Attribute_Directory)
       = File_Attribute_Directory then
    begin
      // Nothing, directory
      SubItems.Add('');
    end
    else
    begin
      // File size
      SubItems.Add(FileSizeFormat(AData.nFileSizeLow));
    end;

    SubItems.Add(FormatDateTime('dd/mm/yyyy hh:mm:ss',
                 FileTimeToDateTime(AData.ftLastWriteTime)));
  end;
end;

{------------------------------------------------------------------------------}

function TFMain.GetListItemType(Item: TListItem): TListItemType;
begin
  if not assigned(Item.Data) then result := itFile
  else
  begin
    if (TWin32FindData(Item.Data^).dwFileAttributes and faDirectory <> 0) then
      result := itFolder
    else
      result := itFile;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.ClearLanguageMenu;
// Clear the language menu
var
  N: integer;
begin
  if MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex].Count > 0 then
  begin
    for N:= MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex].Count-1 downto 0 do
      MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex].Items[N].Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.MnuLanguageClick(Sender: TObject);
// Change language
var
  ALanguage: string;
  APos: integer;
begin
  with Sender as TMenuItem do
  begin
    ALanguage:= ClearShortcutText(Caption);
    with MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex] do
    begin
      for APos:= 1 to Count do
        Items[APos-1].Checked:= false;
    end;
    Checked:= true;
    LoadLanguage(ALanguage);
    AppliCfg.Language:= ALanguage;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.BuildLanguageMenu;
// Build the language menu
var
  ASearchRec: TSearchRec;
  AText: string;
  Number: integer;
  AMenuItem: TMenuItem;
begin
  Number:= 0;
  if FindFirst(AppliDir + '*' + LngExt, faAnyFile, ASearchRec) = 0 then
  begin
    ClearLanguageMenu;
    inc(Number);
    AText:= ExtractFileName(ASearchRec.Name);
    AText:= copy(AText,1,length(AText)- length(LngExt));
    AText:= FirstUpperText(AText);
    // Add sub menu
    AMenuItem:= TMenuItem.Create(MnuMain);
    AMenuItem.Caption:= AText;
    AMenuItem.OnClick:= MnuLanguageClick;
    MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex].Add(AMenuItem);
    // Find next language
    while FindNext(ASearchRec) = 0 do
    begin
      inc(Number);
      AText:= ExtractFileName(ASearchRec.Name);
      AText:= copy(AText,1,length(AText)-4);
      AText:= FirstUpperText(AText);
      // Add sub menu
      AMenuItem:= TMenuItem.Create(MnuMain);
      AMenuItem.Caption:= AText;
      AMenuItem.OnClick:= MnuLanguageClick;
      MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex].Add(AMenuItem);
    end;
  end;
  with MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex] do
    if Number < 2 then Visible:= false
                  else Visible:= true;
  if Number=0 then CreateLanguageFile;
end;

{------------------------------------------------------------------------------}

procedure TFMain.InitAppli;
var
  N: integer;
begin
  // Set language
  LoadLanguage(AppliCfg.Language);
  with MnuMain.Items[ToolMnuIndex].Items[LanguageMnuIndex] do
  begin
    for N:= 0 to pred(Count) do
    begin
      if ClearShortcutText(Items[N].Caption) = AppliCfg.Language then
         Items[N].Checked:= true
      else Items[N].Checked:= false;
    end;
  end;
  // Load serial configuration list (Machine)
  GetAllConfigName(SerialCfgFilename,CbxMachine.Items);
  with AppliCfg do
  begin
    // Load selected serial configuration
    if Machine = '' then Machine:= CbxMachine.Items[0];
    CbxMachine.ItemIndex:= CbxMachine.Items.IndexOf(AppliCfg.Machine);
    ActChangeMachine.Execute;
    // Load window geometry
    if SaveMainWindow then
    begin
      // Be sure to be visible
      if (MainWindow.Left > Screen.Width) or (MainWindow.Right < 0) or
         (MainWindow.Top > Screen.Height) or (MainWindow.Bottom < 0) then
         Position:= PoScreenCenter
      else
      begin
        Position:= poDesigned;
        BoundsRect:= MainWindow;
      end;
    end
    else Position:= poScreenCenter;
    // Change File list display
    case FileListDisplay of
      0: ActSmallIcon.Execute;
      1: ActList.Execute;
      2: ActDetail.Execute;
    else
      LvFileList.ViewStyle:= vsSmallIcon;
    end;
    // Display info panel
    PnlInfo.Visible:= InfoSignals;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.GotoDir(ADirectory: string);
begin
  ADirectory:= AddBackSlash(ADirectory);
  if DirectoryExists(ADirectory) then
  begin
    CurrentDir:= ADirectory;
    CurrentDrive:= (lowercase(CurrentDir))[1];
    DcbPCDriveList.Drive:= CurrentDrive;
  end;
  ActReadDir.Execute;
end;

{------------------------------------------------------------------------------}

procedure TFMain.UpdateGUI(Transmitting: boolean);
begin
  MnuTools.Enabled:= not Transmitting;
  TbrPC.Enabled:= not Transmitting;

  CbxMachine.Enabled:= not Transmitting;
  SbnConfig.Enabled:= not Transmitting;
  ActSend.Enabled:= not Transmitting;
  ActReceive.Enabled:= not Transmitting;
  ActCancel.Enabled:= Transmitting;
  ActDisconnect.Visible:= (not Transmitting) and (SerialCom.ComPort.Connected);

  SerialCom.NbTransmittedChar:= 0;
  LblTransmittedChar.Caption:= '';
  TmrTransmittedChar.Enabled:= Transmitting;
end;

{------------------------------------------------------------------------------}

procedure TFMain.CleanMemFollowup;
const
  MaxLines = 250;
var
  N: integer;
begin
  if MemFollowup.Lines.Count > MaxLines then
  begin
    MemFollowup.Lines.BeginUpdate;
    for N:= 1 to 50 do MemFollowup.Lines.Delete(0);
    MemFollowup.Lines.EndUpdate;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFMain.SetLogCfg;
begin
    Log.Enabled:= AppliCfg.LogFile;
    Log.LogItems:= [];
    if AppliCfg.LogStatus then Log.LogItems:= Log.LogItems + [itlStatus];
    if AppliCfg.LogTiming then Log.LogItems:= Log.LogItems + [itlTiming];
    if AppliCfg.LogCode then Log.LogItems:= Log.LogItems + [itlCode];
    if AppliCfg.LogData then Log.LogItems:= Log.LogItems + [itlData];
end;

{------------------------------------------------------------------------------}

procedure TFMain.SerialStatusChange(Sender: TObject; StatusCode: TStatusCode);
{ Process serial communication status changes }
var
  ALogText: string;
  AStatusText: string;
begin
  ALogText:= '';
  AStatusText:= '';
  case StatusCode of
    scConnected         : begin
                            ALogText:= 'Connected';
                          end;
    scDisconnected      : begin
                            ALogText:= 'Disconnected';
                            ActDisconnect.Visible:= false;
                          end;
    scBufferFull        : begin
                            ALogText:= 'Buffer full';
                          end;
    scEndOfTransmission : begin
                            ALogText:= 'End of transmission';
                            Log.Save;
                          end;
    scSendFile          : begin
                            ALogText:= 'Sending file: ' + CurrentFilename;
                          end;
    scReceiveFile       : begin
                            ALogText:= 'Receiving file: ' + CurrentFilename;
                          end;
    scAskForData        : begin
                            ALogText:= 'Requesting data';
                          end;
    scWaitForSending    : begin
                            ALogText:= 'Waiting code for sending';
                          end;
    scGetStartToSend    : begin
                            ALogText:= 'Received start to send signal';
                          end;
    scGetXOFF           : begin
                            ALogText:= 'Received XOFF';
                          end;
    scGetXON            : begin
                            ALogText:= 'Received  XON';
                          end;
    scSendXOFF          : begin
                            ALogText:= 'Send XOFF';
                          end;
    scSendXON           : begin
                            ALogText:= 'Send  XON';
                          end;
  end;
  if ALogText <> '' then
    Log.Write(timeGetTime - StartTime, 0, ALogText, ltStatus);
end;

{------------------------------------------------------------------------------}

procedure TFMain.SerialComProcessRxCode(Sender: TObject; Code: Integer);
{ Log received code at processing time }
begin
  Log.Write(timeGetTime - StartTime, Code, '', ltRxCode);
end;

{------------------------------------------------------------------------------}

procedure TFMain.SerialComProcessTxCode(Sender: TObject; Code: Integer);
{ Log transmitted code at processing time }
begin
  Log.Write(timeGetTime - StartTime, Code, '', ltTxCode);
end;

{------------------------------------------------------------------------------}

procedure TFMain.SerialComProcessRxValue(Sender: TObject; Value: Integer);
{ Log received value at processing time }
begin
  Log.Write(timeGetTime - StartTime, Value, '', ltRxValue);
end;

{------------------------------------------------------------------------------}

procedure TFMain.SerialComProcessTxValue(Sender: TObject; Value: Integer);
{ Log transmitted value at processing time }
begin
  Log.Write(timeGetTime - StartTime, Value, '', ltTxValue);
end;


{*****************************************************************************}
{ Public function                                                             }
{*****************************************************************************}

procedure TFMain.LvFileListMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  P: TPoint;
  AnItem: TListItem;
begin
  if AppliCfg.Thumbnails then
  begin
    P := LvFileList.ScreenToClient(Mouse.CursorPos) ;
    AnItem := LvFileList.GetItemAt(P.x, P.y);
    if AnItem = nil then
      FThumb.Hide
    else
    begin
      FThumb.Left:= Left + PnlMachine.Left + 10;
      FThumb.Top:= Top + PnlPCFileList.Top + 50;
      FThumb.Display(CurrentDir, AnItem.Caption);
    end;
  end
  else FThumb.Hide;
end;

end.
