unit RSLog;

/////////////////////////////////////////////////////////////////////////////
//                     Part of RS232 project                               //
//        Sending and receiving files through RS232 interface              //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Common element for that project                                         //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

interface

uses Classes, SysUtils;

type
  TLogType = (ltStatus, ltTxCode, ltTxValue, ltRxCode, ltRxValue, ltInfo);
  TItemToLog = (itlStatus, itlTiming, itlCode, itlData);
  TITLArray = set of TItemToLog;

  TRSLog = class(TComponent)
  private
    FFilename: TFilename;
    FLogFile: textfile;
    FEnabled: boolean;
    FLogItems: TITLArray;
    FLogList: TStringList;
    function RightEnlarge(AText: string; ALength: byte): string;
    function MillisecondesToStr(ATiming: longword): string;
    function CodeToStr(ACode: byte): string;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Filename: TFilename read FFilename write FFilename;
    property Enabled: boolean read FEnabled write FEnabled;
    property LogItems: TITLArray read FLogItems write FLogItems;
    function Reset: boolean;
    function Save: boolean;
    procedure Write(ATiming: longword; AValue: byte; AText: string;
      AType: TLogType);
  end;

var
  Log: TRSLog;

implementation

uses
  RSCommon;

{ TRSLog }

{------------------------------------------------------------------------------}
{ Private                                                                      }
{------------------------------------------------------------------------------}

function TRSLog.RightEnlarge(AText: string; ALength: byte): string;
{ Enlarge to the right or cut at the specified size }
begin
  if length(AText)>= ALength then
  begin
    Result:= copy(AText,1,ALength);
  end
  else
  begin
    while length(AText) < ALength do AText:= AText + ' ';
    Result:= AText;
  end;
end;

{------------------------------------------------------------------------------}

function TRSLog.MillisecondesToStr(ATiming: longword): string;
{ Convert millisecondes to an HH:MM:SS.mmm string }
var
  Hours: word;
  Minutes: word;
  Secondes: word;
  Millisecondes: word;
begin
  Hours:= ATiming div 3600000;
  if Hours <> 0 then ATiming:= ATiming mod 3600000;
  Minutes:= ATiming div 60000;
  if Minutes <> 0 then ATiming:= ATiming mod 60000;
  Secondes:= ATiming div 1000;
  Millisecondes:= ATiming mod 1000;
  Result:= FormatFloat('00',Hours) + ':';
  Result:= Result + FormatFloat('00',Minutes) + ':';
  Result:= Result + FormatFloat('00',Secondes) + '.';
  Result:= Result + FormatFloat('000',Millisecondes);
end;

{------------------------------------------------------------------------------}

function TRSLog.CodeToStr(ACode: byte): string;
{ Return string corresponding to a code }
begin
  case ACode of
    00: Result:= 'NUL';
    01: Result:= 'SOH';
    02: Result:= 'STX';
    03: Result:= 'ETX';
    04: Result:= 'EOT';
    05: Result:= 'ENQ';
    06: Result:= 'ACK';
    07: Result:= 'BEL';
    08: Result:= 'BS';
    09: Result:= 'TAB';
    10: Result:= 'LF';
    11: Result:= 'VT';
    12: Result:= 'FF';
    13: Result:= 'CR';
    14: Result:= 'SO';
    15: Result:= 'SI';
    16: Result:= 'DLE';
    17: Result:= 'DC1';
    18: Result:= 'DC2';
    19: Result:= 'DC3';
    20: Result:= 'DC4';
    21: Result:= 'NAK';
    22: Result:= 'SYN';
    23: Result:= 'ETB';
    24: Result:= 'CAN';
    25: Result:= 'EM';
    26: Result:= 'SUB';
    27: Result:= 'ESC';
    28: Result:= 'FS';
    29: Result:= 'GS';
    30: Result:= 'RS';
    31: Result:= 'US';
  else
    Result:= '';
  end;
end;

{------------------------------------------------------------------------------}
{ Public                                                                       }
{------------------------------------------------------------------------------}

constructor TRSLog.Create(AOwner: TComponent);
begin
  inherited;
  FFilename:= '';
  FEnabled:= true;
  FLogItems:= [];
  FLogList:= TStringList.Create;
end;

{------------------------------------------------------------------------------}

destructor TRSLog.Destroy;
begin
  FLogList.Free;
  inherited;
end;

{------------------------------------------------------------------------------}
{ Published                                                                    }
{------------------------------------------------------------------------------}

function TRSLog.Reset: boolean;
{ Create new log file. Delete previous one if exist }
begin
  Result:= false;
  if FEnabled and (FFilename <> '') then
  begin
    FLogList.Clear;
    assignfile(FLogFile,FFilename);
    {$IOCHECKS OFF}
    rewrite(FLogFile);
    writeln(FLogFile, 'RS232 ' + SoftVersion + ' log file' );
    flush(FLogFile);
    closefile(FLogFile);
    {$IOCHECKS ON}
    Result:= IOResult = 0;
  end;
end;

{------------------------------------------------------------------------------}

function TRSLog.Save: boolean;
{ Write current the LogList in the log file }
var
  N: integer;
  ALine: string;
begin
  Result:= true;
  if FEnabled and (FFilename <> '') and (FLogList.Count > 0) then
  begin
    assignfile(FLogFile,FFilename);
    {$IOCHECKS OFF}
    if not FileExists(FFilename) then
    begin
      rewrite(FLogFile);
    writeln(FLogFile, 'RS232 ' + SoftVersion + ' log file' );
    end
    else append(FLogFile);
    for N:= 0 to (FLogList.Count - 1) do
    begin
      ALine:= FLogList.Strings[N];
      writeln(FLogFile, ALine);
    end;
    flush(FLogFile);
    closefile(FLogFile);
    {$IOCHECKS ON}
    FLogList.Clear;
    Result:= IOResult = 0;
    if Result <> true then beep;
  end
end;

{------------------------------------------------------------------------------}

procedure TRSLog.Write(ATiming: longword; AValue: byte; AText: string;
  AType: TLogType);
{ Write a line in the log file }
var
  TimingStr: string;
  ALine: string;
begin
  if FEnabled then
  begin
    ALine:= '';
    if itlTiming in FLogItems then TimingStr:= #9 + MillisecondesToStr(ATiming) + #9
                              else TimingStr:= #9;
    case AType of
      ltStatus : if itlStatus in FLogItems then ALine:= AText;
      ltTxCode : if itlCode in FLogItems then
                 begin
                   ALine:= TimingStr + 'TX' + #9;
                   ALine:= ALine + RightEnlarge(CodeToStr(AValue), 3) + #9;
                   ALine:= ALine + FormatFloat('000',AValue) + #9;
                   ALine:= ALine + '$'+IntToHex(AValue,2) + #9;
                 end;
      ltTxValue: if itlData in FLogItems then
                 begin
                   ALine:= TimingStr + 'TX' + #9;
                   ALine:= ALine + RightEnlarge('', 3) + #9;
                   ALine:= ALine + FormatFloat('000',AValue) + #9;
                   ALine:= ALine + '$'+IntToHex(AValue,2) + #9;
                   if AValue >= 32 then
                   begin
                     ALine:= ALine + chr(AValue) + #9;
                   end;
                 end;
      ltRxCode : if itlCode in FLogItems then
                 begin
                   ALine:= TimingStr + 'RX' + #9;
                   ALine:= ALine + RightEnlarge(CodeToStr(AValue), 3) + #9;
                   ALine:= ALine + FormatFloat('000',AValue) + #9;
                   ALine:= ALine + '$'+IntToHex(AValue,2) + #9;
                 end;
      ltRxValue: if itlData in FLogItems then
                 begin
                   ALine:= TimingStr + 'RX' + #9;
                   ALine:= ALine + RightEnlarge('', 3) + #9;
                   ALine:= ALine + FormatFloat('000',AValue) + #9;
                   ALine:= ALine + '$'+IntToHex(AValue,2) + #9;
                   if AValue >= 32 then
                   begin
                     ALine:= ALine + chr(AValue) + #9;
                   end;
                 end;
      ltInfo   : ALine:= AText;
    end;
    if ALine <> '' then FLogList.Add(ALine);
  end;
end;

end.
