unit RSCommon;

/////////////////////////////////////////////////////////////////////////////
//                     Part of RS232 project                               //
//        Sending and receiving files through RS232 interface              //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Common element for that project                                         //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

interface

uses
  Windows, SysUtils, Types, ShellApi,ShlObj, IniFiles;

const
  SoftVersion = '1.2.3';    // Software version

  RefLng = 'francais';      // Default language
  LngExt = '.lng';          // File extension used with language files
  HelpExt = '.hlp';         // File extension used with help files
  SerialCfgExt = '.crs';    // File extension used by serial configuration file
  ExportCfgExt = '.xrs';    // File extension used by export serial configuration file
  RefSerialCfg = 'Default'; // Default serial configuration

  FileMnuIndex = 0;
  ToolMnuIndex = 1;
  HelpMnuIndex = 2;

  LanguageMnuIndex = 1;

  ThumbDefaultWidth = 300;
  ThumbDefaultHeight = 200;
type
  TAppliCfg = class             // Application configuration
  private
    FSaveLanguage: boolean;         // Save language in use
    FSaveMachine: boolean;          // Save machine in use
    FSaveMainWindow: boolean;       // Save Windows position and size
    FSaveFileListDisplay: boolean;  // Save file list display type
    FLanguage: string;              // Language in use
    FMachine: string;               // Machine in use
    FMainWindow: TRect;             // Window geometry
    FFileListDisplay: integer;      // Kind of file list display (icon, list, detail)
    FLogFile: boolean;              // Did we make log file
    FLogStatus: boolean;            // Did we log status
    FLogTiming: boolean;            // Did we log timing
    FLogCode: boolean;              // Did we log codes
    FLogData: boolean;              // Did we log data
    FInfoSignals: boolean;          // Did we display signals leds
    FThumbnails: boolean;           // Did we display thumbnails
    FThumbWidth: integer;           // Thumbnail width
    FThumbHeight: integer;          // Thumbnail height
  public
    Constructor Create;
    Destructor Destroy; Override;
    procedure CreateDefault(AFilename: TFilename);
    function LoadFromFile(AFilename: TFilename): boolean;
    function SaveToFile(AFilename: TFilename): boolean;
  published
    property SaveLanguage: boolean read FSaveLanguage write FSaveLanguage;
    property SaveMachine: boolean read FSaveMachine write FSaveMachine;
    property SaveMainWindow: boolean read FSaveMainWindow write FSaveMainWindow;
    property SaveFileListDisplay: boolean
      read FSaveFileListDisplay write FSaveFileListDisplay;
    property Language: string read FLanguage write FLanguage;
    property Machine: string read FMachine write FMachine;
    property MainWindow: TRect read FMainWindow write FMainWindow;
    property FileListDisplay: integer read FFileListDisplay write FFileListDisplay;
    property LogFile: boolean read FLogFile write FLogFile;
    property LogStatus: boolean read FLogStatus write FLogStatus;
    property LogTiming: boolean read FLogTiming write FLogTiming;
    property LogCode: boolean read FLogCode write FLogCode;
    property LogData: boolean read FLogData write FLogData;
    property InfoSignals: boolean read FInfoSignals write FInfoSignals;
    property Thumbnails: boolean read FThumbnails write FThumbnails;
    property ThumbWidth: integer read FThumbWidth write FThumbWidth;
    property ThumbHeight: integer read FThumbHeight write FThumbHeight;
  end;

var
  AppliDir: TFilename;           // Directory of exe
  AppliCfgDir: TFilename;        // Directory of configuration files
  AppliCfgFilename: TFilename;   // Application configuration filename
  AppliCfg: TAppliCfg;           // Application configuration
  SerialCfgFilename: TFilename;  // Serial communication configuration filename

  CurrentDrive: char;            // Current drive on PC
  CurrentDir: string;            // Current directory on PC

  function AddBackSlash(AString: string): string;
  function FileSizeFormat(Bytes: longword): string;
  function FileTimeToDateTime(AFileTime: TFileTime): TDateTime;
  function PCPreviousDir(ADirectory: string):string;
  function GetShellInfo(const AFileName: string;
    dwFileAttributes: DWORD): TSHFileInfo;
  function GetSpecialFolder(Folder: Integer): String;
  function GetAppdataFolder: string;

  function LowerCaseText(AText: string): string;
  function FirstUpperText(AText: string): string;
  function ClearShortcutText(AText: string): string;

  function CheckNumValue(AText: string;Min,Max: integer): boolean;
implementation

{*****************************************************************************}
{ Utilities                                                                   }
{*****************************************************************************}

function AddBackSlash(AString: string): string;
// Add an ending backslash if needed
begin
  if copy(AString,length(AString),1) <> '\' then result:= AString + '\'
                                            else result := AString;
end;

{------------------------------------------------------------------------------}

function FileSizeFormat(Bytes: dword): string;
// Format passed file size
begin
  result:= format('%d %s', [Bytes, '']);
end;

{------------------------------------------------------------------------------}

function  FileTimeToDateTime(AFileTime: TFileTime): TDateTime;
{
  Snaffled from UseNet, so don't whine about the hard-coded magic numbers.
  It does what it's supposed to :)
}
begin
  result := (Comp(AFileTime) / 8.64E11) - 109205.0
end;

{------------------------------------------------------------------------------}

function PCPreviousDir(ADirectory: string):string;
// Return previous dir in the tree
var
  DirLength: integer;
begin
  DirLength:= length(ADirectory);
  // Remove char after '\'
  while (copy(ADirectory,DirLength,1)<>'\') and (DirLength>0) do
  begin
    delete(ADirectory,DirLength,1);
    dec(DirLength);
  end;
  // Remove '\'
  if DirLength>0 then
  begin
    delete(ADirectory,DirLength,1);
    dec(DirLength);
  end;
  // Find previous '\'
  while (copy(ADirectory,DirLength,1)<>'\') and (DirLength>0) do
  begin
    delete(ADirectory,DirLength,1);
    dec(DirLength);
  end;
  result:= ADirectory;
end;

{------------------------------------------------------------------------------}

function GetShellInfo(const AFileName: string;
  dwFileAttributes: DWORD): TSHFileInfo;
begin
  SHGetFileInfo(PChar(AFileName), dwFileAttributes, result, SizeOf(TSHFileInfo),
    SHGFI_SYSICONINDEX or SHGFI_USEFILEATTRIBUTES);
end;

{------------------------------------------------------------------------------}

function GetSpecialFolder(Folder: Integer): String;
// Function to get special folders.
// Application data: CSIDL_APPDATA
// Local application data: CSIDL_LOCAL_APPDATA
// Common application data: CSIDL_COMMON_APPDATA
// Common document: CSIDL_COMMON_DOCUMENTS
// My documents: CSIDL_PERSONAL

var
  SFolder : pItemIDList;
  SpecialPath : Array[0..MAX_PATH] Of Char;
begin
  SHGetSpecialFolderLocation(0, Folder, SFolder);
  SHGetPathFromIDList(SFolder, SpecialPath);
  Result := StrPas(SpecialPath);
end;

{------------------------------------------------------------------------------}

function GetAppdataFolder: string;
// Function to get the app data special folder.
begin
   Result := GetSpecialFolder(CSIDL_APPDATA);
end;

{------------------------------------------------------------------------------}

function LowerCaseText(AText: string): string;
// Change all to lower case
begin
  result:= AnsiLowerCase(AText);
end;

{------------------------------------------------------------------------------}

function FirstUpperText(AText: string): string;
// Put first character in upper case, other lower case
var
  FirstChar: string;
begin
  AText:= LowerCaseText(AText);
  FirstChar:= AText[1];
  FirstChar:= AnsiUpperCase(FirstChar);
  system.Delete(AText,1,1);
  result:= FirstChar + AText;
end;

{------------------------------------------------------------------------------}

function ClearShortcutText(AText: string): string;
// Remove '&' corresponding to shortcut identification and change to lower case
var
  APos: integer;
begin
  AText:= AnsiLowerCase(AText);
  APos:= pos('&',AText);
  if APos<>0 then System.delete(AText,APos,1);
  result:= AText;
end;

{*****************************************************************************}
{ Application configuration                                                   }
{*****************************************************************************}

constructor TAppliCfg.Create;
// Create application configuration
begin
  inherited;
end;

{------------------------------------------------------------------------------}

destructor TAppliCfg.Destroy;
// Destroy application configuration
begin
  inherited;
end;

{------------------------------------------------------------------------------}

procedure TAppliCfg.CreateDefault(AFilename: TFilename);
// Create a default application configuration file
begin
  // Preference
  SaveLanguage:= true;
  SaveMachine:= true;
  SaveMainWindow:= false;
  SaveFileListDisplay:= true;
  // Values
  Language:= RefLng;
  Machine:= '';
  MainWindow:= Bounds(0,0,640,480);
  FileListDisplay:= 0;
  LogFile:= false;
  LogStatus:= false;
  LogTiming:= false;
  LogCode:= false;
  LogData:= false;
  InfoSignals:= false;
  Thumbnails:= false;
  ThumbWidth:= ThumbDefaultWidth;
  ThumbHeight:= ThumbDefaultHeight;
  // Save it
  SaveToFile(AFilename);
end;

{------------------------------------------------------------------------------}

function TAppliCfg.LoadFromFile(AFilename: TFilename): boolean;
// Load application configuration from a file
var
  ConfigFile: TIniFile;
begin
  result:= true;
  if FileExists(AFilename) then
  begin
    // Open configuration file
    ConfigFile:= TIniFile.Create(AFilename);
    // Load preferences
    SaveLanguage:= ConfigFile.ReadBool('Preference','Language',true);
    SaveMachine:= ConfigFile.ReadBool('Preference','Machine',true);
    SaveMainWindow:= ConfigFile.ReadBool('Preference','Window',false);
    SaveFileListDisplay:= ConfigFile.ReadBool('Preference','FileListDisplay',true);
    // Load values
    if SaveLanguage then
      Language:= ConfigFile.ReadString('Values','Language',RefLng)
    else Language:= RefLng;
    if SaveMachine then
      Machine:= ConfigFile.ReadString('Values','Machine',RefSerialCfg)
    else Machine:= RefSerialCfg;
    if SaveMainWindow then
      MainWindow:= Rect(
        ConfigFile.ReadInteger('Values','WindowLeft',0),
        ConfigFile.ReadInteger('Values','WindowTop',0),
        ConfigFile.ReadInteger('Values','WindowRight',640),
        ConfigFile.ReadInteger('Values','WindowBottom',480)
      )
    else MainWindow:= Rect(0,0,0,0);
    if SaveFileListDisplay then
      FileListDisplay:= ConfigFile.ReadInteger('Values','FileListDisplay',0)
    else FileListDisplay:= 0;
    LogFile:= ConfigFile.ReadBool('Preference','LogFile',false);
    LogStatus:= ConfigFile.ReadBool('Preference','LogStatus',false);
    LogTiming:= ConfigFile.ReadBool('Preference','LogTiming',false);
    LogCode:= ConfigFile.ReadBool('Preference','LogCode',false);
    LogData:= ConfigFile.ReadBool('Preference','LogData',false);
    InfoSignals:= ConfigFile.ReadBool('Preference','InfoSignals',false);
    Thumbnails:= ConfigFile.ReadBool('Preference','Thumbnails',false);
    ThumbWidth := ConfigFile.ReadInteger('Preference','ThumbWidth',ThumbDefaultWidth);
    ThumbHeight := ConfigFile.ReadInteger('Preference','ThumbHeight',ThumbDefaultHeight);
    // Close configuration file
    ConfigFile.Free;
  end
  else CreateDefault(AFilename);
end;

{------------------------------------------------------------------------------}

function TAppliCfg.SaveToFile(AFilename: TFilename): boolean;
// Save application configuration to a file
var
  ConfigFile: TIniFile;
begin
  result:= true;
  // Create, open configuration file
  ConfigFile:= TIniFile.Create(AFilename);
  // Save preferences
  ConfigFile.WriteBool('Preference','Language',SaveLanguage);
  ConfigFile.WriteBool('Preference','Machine',SaveMachine);
  ConfigFile.WriteBool('Preference','Window',SaveMainWindow);
  ConfigFile.WriteBool('Preference','FileListDisplay',SaveFileListDisplay);
  // Save values
  if SaveLanguage then ConfigFile.WriteString('Values','Language',Language);
  if SaveMachine then  ConfigFile.WriteString('Values','Machine',Machine);
  if SaveMainWindow then
  begin
    ConfigFile.WriteInteger('Values','WindowLeft',MainWindow.Left);
    ConfigFile.WriteInteger('Values','WindowTop',MainWindow.Top);
    ConfigFile.WriteInteger('Values','WindowRight',MainWindow.Right);
    ConfigFile.WriteInteger('Values','WindowBottom',MainWindow.Bottom);
  end;
  if SaveFileListDisplay then
    ConfigFile.WriteInteger('Values','FileListDisplay',FileListDisplay);
  ConfigFile.WriteBool('Preference','LogFile',LogFile);
  ConfigFile.WriteBool('Preference','LogStatus',LogStatus);
  ConfigFile.WriteBool('Preference','LogTiming',LogTiming);
  ConfigFile.WriteBool('Preference','LogCode',LogCode);
  ConfigFile.WriteBool('Preference','LogData',LogData);
  ConfigFile.WriteBool('Preference','InfoSignals',InfoSignals);
  ConfigFile.WriteBool('Preference','Thumbnails',Thumbnails);
  ConfigFile.WriteInteger('Preference','ThumbWidth',ThumbWidth);
  ConfigFile.WriteInteger('Preference','ThumbHeight',ThumbHeight);
  // Close configuration file
  ConfigFile.Free;
end;

{------------------------------------------------------------------------------}

function CheckNumValue(AText: string;Min,Max: integer): boolean;
var
  Value: longint;
begin
  Result:= true;
  try
    Value:= StrToInt(AText);
    if ((Value < Min) or (Value > Max)) then Result:= false;
  except
    on EConvertError do
    begin
      Result:= false;
    end;
  end;
end;

end.
