unit RSCfgmng;

/////////////////////////////////////////////////////////////////////////////
//                     Part of RS232 project                               //
//        Sending and receiving files through RS232 interface              //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Serial configuration management for that project                        //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA .            }
{***************************************************************************}

interface

uses
  SysUtils, Classes;

const
  // Coding type
  Sans = 0;
  ISO  = 1;
  EIA  = 2;
  // Configuration file version
  VersionStr = 'Version';
  CfgVersion = '1.0';

type
  TCfgSerial=class
  private
    FConfigName: string;
    FWorkingDir: string;
    FStartCode: string;
    FEndCode: string;
    FSendCode: string;
    FCoding: byte;
    FPort: string;
    FSpeed: string;
    FParity: byte;
    FDataBits: string;
    FStopBits: string;
    FFlowControl: byte;
    FStartNullChar: string;
    FEndNullChar: string;
    FMessageBefore: boolean;
    FSendRemoveLines: string;
    FSendFindStartEnd: boolean;
    FSendWaitSignal: boolean;
    FStayOpen: boolean;
    FReceiveRemoveLines: string;
    FReceiveFindStartEnd: boolean;
    FReceiveWaitSignal: boolean;

    procedure ExtractParam(AText: string);
    procedure ExtractComSetting(AText: string);
    procedure ExtractSendSetting(AText: string);
    procedure ExtractReceiveSetting(AText: string);

    function BuildParamLine: string;
    function BuildComSettingLine: string;
    function BuildSendSettingLine: string;
    function BuildReceiveSettingLine: string;

  public
    constructor Create;
    destructor Destroy; Override;
    procedure InitToDefault;
    procedure LoadFromFile(AFilename: TFilename; AName: string);
    procedure SaveToFile(AFilename: TFilename; AName: string);
    procedure DeleteFromFile(AFilename: TFilename; AName: string);
    procedure ExportToFile(AFilename: TFilename; AName: string);
    function ImportFromFile(AFilename: TFilename): boolean;
  published
    property ConfigName: string read FConfigName write FConfigName;
    property WorkingDir: string read FWorkingDir write FWorkingDir;
    property StartCode: string read FStartCode write FStartCode;
    property EndCode: string read FEndCode write FEndCode;
    property SendCode: string read FSendCode write FSendCode;
    property Coding: byte read FCoding write FCoding;
    property Port: string read FPort write FPort;
    property Speed: string read FSpeed write FSpeed;
    property Parity: byte read FParity write FParity;
    property DataBits: string read FDataBits write FDataBits;
    property StopBits: string read FStopBits write FStopBits;
    property FlowControl: byte read FFlowControl write FFlowControl;
    property StartNullChar: string read FStartNullChar write FStartNullChar;
    property EndNullChar: string read FEndNullChar write FEndNullChar;
    property MessageBefore: boolean read FMessageBefore write FMessageBefore;
    property SendRemoveLines: string read FSendRemoveLines write FSendRemoveLines;
    property SendFindStartEnd: boolean read FSendFindStartEnd write FSendFindStartEnd;
    property SendWaitSignal: boolean read FSendWaitSignal write FSendWaitSignal;
    property StayOpen: boolean read FStayOpen write FStayOpen;
    property ReceiveRemoveLines: string read FReceiveRemoveLines write FReceiveRemoveLines;
    property ReceiveFindStartEnd: boolean read FReceiveFindStartEnd write FReceiveFindStartEnd;
    property ReceiveWaitSignal: boolean read FReceiveWaitSignal write FReceiveWaitSignal;
  end;

var
  CurrentCfg: TCfgSerial;
  TmpCfg: TCfgSerial;

  procedure CreateSerialCfgFile;
  procedure GetAllConfigName(AFilename: TFilename; AStringList: TStrings);
  procedure CopySerialCfg(SourceCfg, DestCfg: TCfgSerial);

implementation

uses
  RScommon;

const
  DefaultStartCode           = '';
  DefaultEndCode             = '';
  DefaultSendCode            = '';
  DefaultCoding              = 0;
  DefaultPort                = 'COM1';
  DefaultSpeed               = '9600';
  DefaultParity              = 0;
  DefaultDataBits            = '8';
  DefaultStopBits            = '1';
  DefaultFlowControl         = 0;
  DefaultStartNullChar       = '0';
  DefaultEndNullChar         = '0';
  DefaultMessageBefore       = false;
  DefaultSendRemoveLines     = '0';
  DefaultSendFindStartEnd    = false;
  DefaultSendWaitSignal      = false;
  DefaultStayOpen            = false;
  DefaultReceiveRemoveLines  = '0';
  DefaultReceiveFindStartEnd = false;
  DefaultReceiveWaitSignal   = false;

{*****************************************************************************}
{ Utilities                                                                   }
{*****************************************************************************}

procedure ParseCfgLine(ALine: string;var ANumber: longint; var AText: string);
var
  N: integer;
begin
  N:= pos('#',ALine);
  if N<>0 then
  begin
    ANumber:= StrToInt(copy(ALine,1,N-1));
    delete(ALine,1,N);
    AText:= ALine;
  end
  else
  begin
    ANumber:= 0;
    AText:= '';
  end;
end;

{*****************************************************************************}
{ TCfgSerial                                                                  }
{*****************************************************************************}

procedure TCfgSerial.ExtractParam(AText: string);
var
  APos: integer;
begin
  // Working dir
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      WorkingDir:= copy(AText,1,APos-1);
      if not DirectoryExists(WorkingDir) then WorkingDir:= AppliDir;
    end
    else WorkingDir:= AppliDir;
    delete(AText,1,APos);
    if not DirectoryExists(WorkingDir) then WorkingDir:= AppliDir;
  end
  else WorkingDir:= AppliDir;
  // Coding
  if length(AText) > 0 then
  begin
      APos:= pos(',',AText);
      if APos > 1 then
      begin
        Coding:= StrToInt(copy(AText,1,APos-1));
        delete(AText,1,APos);
      end
      else
      begin
        Coding:= StrToInt(AText);
        AText:= '';
      end;
  end
  else Coding:= DefaultCoding;
  // Start code
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos >= 1 then
    begin
      StartCode:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      StartCode:= AText;
      Atext:= '';
    end;

    if StartCode <> '' then
    begin
      if ((StrToInt(StartCode) < 0) or (StrToInt(StartCode) > 255)) then
        StartCode:= DefaultStartCode;
    end;
  end
  else StartCode:= DefaultStartCode;
  // End code
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos >= 1 then
    begin
      EndCode:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      EndCode:= AText;
      AText:= '';
    end;

    if EndCode <> '' then
    begin
      if ((StrToInt(EndCode) < 0) or (StrToInt(EndCode) > 255)) then
        EndCode:= DefaultEndCode;
    end;
  end
  else EndCode:= DefaultEndCode;
  // Send code
  if length(AText) > 0 then
  begin
    SendCode:= AText;
    if ((StrToInt(SendCode) < 0) or (StrToInt(SendCode) > 255)) then
      SendCode:= DefaultSendCode;
  end
  else SendCode:= DefaultSendCode;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.ExtractComSetting(AText: string);
var
  APos: integer;
begin
  // ComPort
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      Port:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      Port:= AText;
      AText:= '';
    end;
  end
  else Port:= DefaultPort;
  // Speed
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      Speed:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      Speed:= AText;
      AText:= '';
    end;

    if ((StrToInt(Speed) < 0) or (StrToInt(Speed) > 256000)) then
      Speed:= DefaultSpeed;
  end
  else Speed:= DefaultSpeed;
  // Parity
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      Parity:= StrToInt(copy(AText,1,APos-1));
      delete(AText,1,APos);
    end
    else
    begin
      Parity:= StrToInt(AText);
      AText:= '';
    end;

    if Parity > 2 then Parity:= DefaultParity;
  end
  else Parity:= DefaultParity;
  // Data bits
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      DataBits:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      DataBits:= AText;
      AText:= '';
    end;

    if ((StrToInt(DataBits) < 5) or (StrToInt(DataBits) > 8)) then
      DataBits:= DefaultDataBits;
  end
  else DataBits:= DefaultDataBits;
  // Stop bits
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      StopBits:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      StopBits:= AText;
      AText:= '';
    end;

    if not ((StopBits = '1') or (StopBits = '1.5') or (StopBits = '2')) then
      StopBits:= DefaultStopBits;
  end
  else StopBits:= DefaultStopBits;
  // Flow control
  if length(AText) > 0 then
  begin
    FlowControl:= StrToInt(AText);
    if FlowControl > 2 then
      FlowControl:= DefaultFlowControl;
  end
  else FlowControl:= DefaultFlowControl;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.ExtractSendSetting(AText: string);
var
  APos: integer;
begin
  // Message before
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      MessageBefore:= copy(AText,1,APos-1) = '1';
      delete(AText,1,APos);
    end
    else
    begin
      MessageBefore:= AText = '1';
      AText:= '';
    end;
  end
  else MessageBefore:= DefaultMessageBefore;
  // Start null char
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      StartNullChar:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      StartNullChar:= AText;
      AText:= '';
    end;
  end
  else StartNullChar:= DefaultStartNullChar;
  // End null char
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      EndNullChar:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      EndNullChar:= AText;
      AText:= '';
    end;
  end
  else EndNullChar:= DefaultEndNullChar;
  // Send remove lines
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      SendRemoveLines:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      SendRemoveLines:= AText;
      AText:= '';
    end;
  end
  else SendRemoveLines:= DefaultSendRemoveLines;
  // Send find start end
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      SendFindStartEnd:= copy(AText,1,APos-1) = '1';
      delete(AText,1,APos);
    end
    else
    begin
      SendFindStartEnd:= AText = '1';
      AText:= '';
    end;
  end
  else SendFindStartEnd:= DefaultSendFindStartEnd;
  // Send wait signal
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      SendWaitSignal:= copy(AText,1,APos-1) = '1';
      delete(AText,1,APos);
    end
    else
    begin
      SendWaitSignal:= AText= '1';
      AText:= '';
    end;
  end
  else SendWaitSignal:= DefaultSendWaitSignal;
  // Stay open
  if length(AText) > 0 then
  begin
    StayOpen:= AText = '1';
  end
  else StayOpen:= DefaultStayOpen;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.ExtractReceiveSetting(AText: string);
var
  APos: integer;
begin
  // Receive remove lines
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      ReceiveRemoveLines:= copy(AText,1,APos-1);
      delete(AText,1,APos);
    end
    else
    begin
      ReceiveRemoveLines:= AText;
      AText:= '';
    end;
  end
  else ReceiveRemoveLines:= DefaultReceiveRemoveLines;
  // Receive find start end
  if length(AText) > 0 then
  begin
    APos:= pos(',',AText);
    if APos > 1 then
    begin
      ReceiveFindStartEnd:= copy(AText,1,APos-1) = '1';
      delete(AText,1,APos);
    end
    else
    begin
      ReceiveFindStartEnd:= AText = '1';
      AText:= '';
    end;
  end
  else ReceiveFindStartEnd:= DefaultReceiveFindStartEnd;
  // Receive wait signal
  if length(AText) > 0 then
  begin
    ReceiveWaitSignal:= AText = '1';
  end
  else ReceiveWaitSignal:= DefaultReceiveWaitSignal;
end;

{------------------------------------------------------------------------------}

function TCfgSerial.BuildParamLine: string;
begin
  // Working dir
  result:= WorkingDir + ',';
  // Coding
  result:= result + IntToStr(Coding) + ',';
  // Start code
  result:= result + StartCode + ',';
  // End code
  result:= result + EndCode + ',';
  // Send code
  result:= result + SendCode;
end;

{------------------------------------------------------------------------------}

function TCfgSerial.BuildComSettingLine: string;
begin
  // ComPort
  result:= Port + ',';
  // Speed
  result:= result + Speed + ',';
  // Parity
  result:= result + IntToStr(Parity) + ',';
  // Data bits
  result:= result + DataBits + ',';
  // Stop bits
  result:= result + StopBits + ',';
  // Flow control
  result:= result + IntToStr(FlowControl);
end;

{------------------------------------------------------------------------------}

function TCfgSerial.BuildSendSettingLine: string;
begin
  // Message before
  if MessageBefore then result:= '1,'
                   else result:= '0,';
  // Start null char
  result:= result + StartNullChar + ',';
  // End null char
  result:= result + EndNullChar + ',';
  // Send remove lines
  result:= result + SendRemoveLines + ',';
  // Send find start end
  if SendFindStartEnd then result:= result + '1,'
                      else result:= result + '0,';
  // Send Wait Signal
  if SendWaitSignal then result:= result + '1,'
                    else result:= result + '0,';
  // StayOpen
  if StayOpen then result:= result + '1'
              else result:= result + '0';
end;

{------------------------------------------------------------------------------}

function TCfgSerial.BuildReceiveSettingLine: string;
begin
  // Receive remove lines
  result:= ReceiveRemoveLines + ',';
  // Receive find start end
  if ReceiveFindStartEnd then result:= result + '1,'
                         else result:= result + '0,';
  // Receive Wait Signal
  if ReceiveWaitSignal then result:= result + '1'
                       else result:= result + '0';
end;

{------------------------------------------------------------------------------}

constructor TCfgSerial.Create;
begin
  inherited;
end;

{------------------------------------------------------------------------------}

destructor TCfgSerial.Destroy;
begin
  inherited;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.InitToDefault;
begin
    FConfigName:= RefSerialCfg;
    FWorkingDir:= AppliDir;
    FStartCode:= DefaultStartCode;
    FEndCode:= DefaultEndCode;
    FSendCode:= DefaultSendCode;
    FCoding:= DefaultCoding;
    FPort:= DefaultPort;
    FSpeed:= DefaultSpeed;
    FParity:= DefaultParity;
    FDataBits:= DefaultDataBits;
    FStopBits:= DefaultStopBits;
    FFlowControl:= DefaultFlowControl;
    FStartNullChar:= DefaultStartNullChar;
    FEndNullChar:= DefaultEndNullChar;
    FMessageBefore:= DefaultMessageBefore;
    FSendRemoveLines:= DefaultSendRemoveLines;
    FSendFindStartEnd:= DefaultSendFindStartEnd;
    FSendWaitSignal:= DefaultSendWaitSignal;
    FStayOpen:= DefaultStayOpen;
    FReceiveRemoveLines:= DefaultReceiveRemoveLines;
    FReceiveFindStartEnd:= DefaultReceiveFindStartEnd;
    FReceiveWaitSignal:= DefaultReceiveWaitSignal;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.LoadFromFile(AFilename: TFilename; AName: string);
var
  SerialCfgFile: textfile;
  Found: boolean;
  ALine: string;
  ANumber: longint;
  AText: string;
begin
  if AFilename <> '' then
  begin
    AssignFile(SerialCfgFile,AFilename);
    reset(SerialCfgFile);
    Found:= false;

    while not (Eof(SerialCfgFile) or Found) do
    begin
      readln(SerialCfgFile,ALine);
      ALine:= trim(ALine);
      if (ALine<>'') and (ALine[1]<>'{') then
      begin
        ParseCfgLine(ALine,ANumber,AText);
        Found:= (ANumber = 1) and (AText = AName);
      end;
    end;

    if Found then
    begin
      ConfigName:= AName;
      while (Found and (not Eof(SerialCfgFile))) do
      begin
        readln(SerialCfgFile,ALine);
        ALine:= trim(ALine);
        if (ALine<>'') and (ALine[1]<>'{') then
        begin
          ParseCfgLine(ALine,ANumber,AText);
          case ANumber of
            1: Found:= false;
            2: ExtractParam(AText);
            3: ExtractComSetting(AText);
            4: ExtractSendSetting(AText);
            5: ExtractReceiveSetting(AText);
          end;
        end;
      end;
    end;

    closefile(SerialCfgFile);
  end;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.SaveToFile(AFilename: TFilename; AName: string);
var
  SerialCfgFile: textfile;
  OldSerialCfgFilename: string;
  OldSerialCfgFile: textfile;
  Found: boolean;
  ALine: string;
  ANumber: longint;
  AText: string;
begin
  if AFilename <> '' then
  begin
    // Backup serial communication file
    OldSerialCfgFilename:= AppliDir + 'old' + SerialCfgExt;
    if FileExists(OldSerialCfgFilename) then deletefile(OldSerialCfgFilename);
    renamefile(AFilename,OldSerialCfgFilename);
    AssignFile(OldSerialCfgFile,OldSerialCfgFilename);
    reset(OldSerialCfgFile);
    // Create new serial communication file
    AssignFile(SerialCfgFile,AFilename);
    rewrite(SerialCfgFile);
    Found:= false;
    // Find AName
    while not (Eof(OldSerialCfgFile) or Found) do
    begin
      readln(OldSerialCfgFile,ALine);
      ALine:= trim(ALine);
      if ALine[1]='{' then writeln(SerialCfgFile,ALine)
      else
      if ALine<>'' then
      begin
        ParseCfgLine(ALine,ANumber,AText);
        if (ANumber = 1) and (AText = AName) then Found:= true
                                             else writeln(SerialCfgFile,ALine);
      end;
    end;
    // Write serial configuration
    writeln(SerialCfgFile,'1#' + AName);
    writeln(SerialCfgFile,'2#' + BuildParamLine);
    writeln(SerialCfgFile,'3#' + BuildComSettingLine);
    writeln(SerialCfgFile,'4#' + BuildSendSettingLine);
    writeln(SerialCfgFile,'5#' + BuildReceiveSettingLine);
    writeln(SerialCfgFile,'{--------------------------------}');
    // Write other serial configuration
    while not Eof(OldSerialCfgFile) do
    begin
      readln(OldSerialCfgFile,ALine);
      if Found then
      begin
        // Skip line of the replaced one
        if (ALine<>'') and (ALine[1]<>'{') then
        begin
          ParseCfgLine(ALine,ANumber,AText);
          Found:= not (ANumber=1);
        end;
      end;
      if not Found then writeln(SerialCfgFile,ALine);
    end;
    // Close
    closefile(OldSerialCfgFile);
    closefile(SerialCfgFile);
  end;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.DeleteFromFile(AFilename: TFilename; AName: string);
var
  SerialCfgFile: textfile;
  OldSerialCfgFilename: string;
  OldSerialCfgFile: textfile;
  Found: boolean;
  ALine: string;
  ANumber: longint;
  AText: string;
begin
  if AFilename <> '' then
  begin
    // Backup serial communication file
    OldSerialCfgFilename:= AppliDir + 'old' + SerialCfgExt;
    if FileExists(OldSerialCfgFilename) then deletefile(OldSerialCfgFilename);
    renamefile(AFilename,OldSerialCfgFilename);
    AssignFile(OldSerialCfgFile,OldSerialCfgFilename);
    reset(OldSerialCfgFile);
    // Create new serial communication file
    AssignFile(SerialCfgFile,AFilename);
    rewrite(SerialCfgFile);
    Found:= false;
    // Find AName
    while not (Eof(OldSerialCfgFile) or Found) do
    begin
      readln(OldSerialCfgFile,ALine);
      ALine:= trim(ALine);
      if ALine[1]='{' then writeln(SerialCfgFile,ALine)
      else
      if ALine<>'' then
      begin
        ParseCfgLine(ALine,ANumber,AText);
        if (ANumber = 1) and (AText = AName) then Found:= true
                                             else writeln(SerialCfgFile,ALine);
      end;
    end;
    // Write other serial configuration
    while not Eof(OldSerialCfgFile) do
    begin
      readln(OldSerialCfgFile,ALine);
      if Found then
      begin
        // Skip line of the find one
        if (ALine<>'') and (ALine[1]<>'{') then
        begin
          ParseCfgLine(ALine,ANumber,AText);
          Found:= not (ANumber=1);
        end;
      end;
      if not Found then writeln(SerialCfgFile,ALine);
    end;
    // Close
    closefile(OldSerialCfgFile);
    closefile(SerialCfgFile);
  end;
end;

{------------------------------------------------------------------------------}

procedure TCfgSerial.ExportToFile(AFilename: TFilename; AName: string);
var
  SerialCfgFile: textfile;
  CfgVersionStr: string;
begin
  if AFilename <> '' then
  begin
    CfgVersionStr:= VersionStr + ' ' + CfgVersion;
    // Create serial communication export file
    AssignFile(SerialCfgFile,AFilename);
    rewrite(SerialCfgFile);
    // Write serial configuration
    writeln(SerialCfgFile,'{--------------------------------}');
    writeln(SerialCfgFile,'{ Fichier d''export config. RS232 }');
    writeln(SerialCfgFile,'{ ' + CfgVersionStr + '                    }');
    writeln(SerialCfgFile,'{ Ne pas modifier manuellement   }');
    writeln(SerialCfgFile,'{ Do not modify manually         }');
    writeln(SerialCfgFile,'{--------------------------------}');
    writeln(SerialCfgFile,'1#' + AName);
    writeln(SerialCfgFile,'2#' + BuildParamLine);
    writeln(SerialCfgFile,'3#' + BuildComSettingLine);
    writeln(SerialCfgFile,'4#' + BuildSendSettingLine);
    writeln(SerialCfgFile,'5#' + BuildReceiveSettingLine);
    writeln(SerialCfgFile,'{--------------------------------}');

    // Close
    closefile(SerialCfgFile);
  end;
end;

{------------------------------------------------------------------------------}

function TCfgSerial.ImportFromFile(AFilename: TFilename): boolean;
var
  SerialCfgFile: textfile;
  Found: boolean;
  ALine: string;
  ANumber: longint;
  AText: string;
  Version: string;
begin
  Result:= false;
  if AFilename <> '' then
  begin
    AssignFile(SerialCfgFile,AFilename);
    reset(SerialCfgFile);
    Found:= false;
    // Look for version
    Version:= '';
    while not (Eof(SerialCfgFile) or Found) do
    begin
      readln(SerialCfgFile,ALine);
      ALine:= trim(ALine);
      if (ALine[1]='{') and (pos(VersionStr,Aline) > 0)then
      begin
        ALine:= copy(ALine,pos(VersionStr,Aline)+length(VersionStr),length(ALine));
        Version:= trim(copy(ALine,1,length(ALine)-1));
        if (Version = CfgVersion) then
        begin
          // Search config start
          while not (Eof(SerialCfgFile) or Found) do
          begin
            readln(SerialCfgFile,ALine);
            ALine:= trim(ALine);
            if ALine<>'' then
            begin
              ParseCfgLine(ALine,ANumber,AText);
              if (ANumber = 1) then Found:= true;
            end;
          end;
          // Read first config
          if Found then
          begin
            ConfigName:= AText;
            while (Found and (not Eof(SerialCfgFile))) do
            begin
              readln(SerialCfgFile,ALine);
              ALine:= trim(ALine);
              if (ALine<>'') and (ALine[1]<>'{') then
              begin
                ParseCfgLine(ALine,ANumber,AText);
                case ANumber of
                  1: Found:= false;
                  2: ExtractParam(AText);
                  3: ExtractComSetting(AText);
                  4: ExtractSendSetting(AText);
                  5: ExtractReceiveSetting(AText);
                end;
              end;
            end;
          end;
          Result:= true;
        end;
      end;
    end;

    // Close
    closefile(SerialCfgFile);
  end;
end;

{*****************************************************************************}
{ Other functions                                                             }
{*****************************************************************************}

procedure CreateSerialCfgFile;
var
  SerialCfgFile: textfile;
  CfgVersionStr: string;
begin
  if SerialCfgFilename <> '' then
  begin
    CfgVersionStr:= VersionStr + ' ' + CfgVersion;
    AssignFile(SerialCfgFile,SerialCfgFilename);
    rewrite(SerialCfgFile);
    writeln(SerialCfgFile,'{--------------------------------}');
    writeln(SerialCfgFile,'{ Fichier de configuration RS232 }');
    writeln(SerialCfgFile,'{ ' + CfgVersionStr + '                    }');
    writeln(SerialCfgFile,'{ Ne pas modifier manuellement   }');
    writeln(SerialCfgFile,'{ Do not modify manually         }');
    writeln(SerialCfgFile,'{--------------------------------}');
    writeln(SerialCfgFile,'1#' + RefSerialCfg);
    writeln(SerialCfgFile,'2#' + AppliDir + ',0,,');
    writeln(SerialCfgFile,'3#COM1,9600,0,8,1,0');
    writeln(SerialCfgFile,'4#0,0,0,0,0,0');
    writeln(SerialCfgFile,'5#0,0,0');

    // Close
    closefile(SerialCfgFile);
  end;

end;

{------------------------------------------------------------------------------}

procedure GetAllConfigName(AFilename: TFilename; AStringList: TStrings);
var
  SerialCfgFile: textfile;
  ALine: string;
  ANumber: longint;
  AText: string;
begin
  if SerialCfgFilename <> '' then
  begin
    AssignFile(SerialCfgFile,SerialCfgFilename);
    reset(SerialCfgFile);
    AStringList.Clear;

    while not Eof(SerialCfgFile) do
    begin
      readln(SerialCfgFile,ALine);
      ALine:= trim(ALine);
      if (ALine<>'') and (ALine[1]<>'{') then
      begin
        ParseCfgLine(ALine,ANumber,AText);
        if ANumber=1 then AStringList.Add(AText);
      end;
    end;

    // Close
    closefile(SerialCfgFile);
  end;
end;

{------------------------------------------------------------------------------}

procedure CopySerialCfg(SourceCfg, DestCfg: TCfgSerial);
begin
  DestCfg.ConfigName:= SourceCfg.ConfigName;
  DestCfg.WorkingDir:= SourceCfg.WorkingDir;
  DestCfg.StartCode:= SourceCfg.StartCode;
  DestCfg.EndCode:= SourceCfg.EndCode;
  DestCfg.SendCode:= SourceCfg.SendCode;
  DestCfg.Coding:= SourceCfg.Coding;
  DestCfg.Port:= SourceCfg.Port;
  DestCfg.Speed:= SourceCfg.Speed;
  DestCfg.Parity:= SourceCfg.Parity;
  DestCfg.DataBits:= SourceCfg.DataBits;
  DestCfg.StopBits:= SourceCfg.StopBits;
  DestCfg.FlowControl:= SourceCfg.FlowControl;
  DestCfg.StartNullChar:= SourceCfg.StartNullChar;
  DestCfg.EndNullChar:= SourceCfg.EndNullChar;
  DestCfg.MessageBefore:= SourceCfg.MessageBefore;
  DestCfg.SendRemoveLines:= SourceCfg.SendRemoveLines;
  DestCfg.SendFindStartEnd:= SourceCfg.SendFindStartEnd;
  DestCfg.SendWaitSignal:= SourceCfg.SendWaitSignal;
  DestCfg.StayOpen:= SourceCfg.StayOpen;
  DestCfg.ReceiveRemoveLines:= SourceCfg.ReceiveRemoveLines;
  DestCfg.ReceiveFindStartEnd:= SourceCfg.ReceiveFindStartEnd;
  DestCfg.ReceiveWaitSignal:= SourceCfg.ReceiveWaitSignal;
end;

end.
